import React from 'react';

import { isObjectEqual } from '../../utils/isObjectEqual';
import { ChatNotification } from '../ChatsCommon/ChatNotification';
import { ARCHIVE_TITLE, BASIC_MENU_TYPES, DEFER_TITLE } from '../ChatsCommon/constants';
import { LinesList } from '../ChatsCommon/LinesList/component';
import { _CHATS, ADDITIONAL_CONTROL_TYPES, ISelectedMenus, LinesMenuType } from '../ChatsCommon/types';
import { FAV_LINES_TYPES, FavouritesLines } from '../FavouritesLines';
import FavouriteLinesConfirm from './FavouriteLinesConfirm';

interface IDispatcherLinesFetcherProps {
    data: any;
    selectedItems: ISelectedMenus;
    onSelect: () => void;
    getArchive: () => void;
    getDeferred: () => void;
    additionalMenuData: any;
    getData: () => void;
    tasksIsLoading: boolean;
}

interface IDispatcherLinesFetcherState {
    linesListData: LinesMenuType;
    favGroup: _CHATS;
}

export default class DispatcherLinesFetcher extends
    React.Component<IDispatcherLinesFetcherProps, IDispatcherLinesFetcherState> {
    state: IDispatcherLinesFetcherState = {
        linesListData: {} as LinesMenuType,
        favGroup: '' as _CHATS,
    };
    favLines = new FavouritesLines();
    chatNotification = new ChatNotification();
    favItem: any = null;

    componentDidMount() {
        if (this.props.data) {
            this.setLinesListData();
        }
    }

    componentDidUpdate(
        prevProps: Readonly<IDispatcherLinesFetcherProps>,
        prevState: Readonly<IDispatcherLinesFetcherState>,
    ) {
        const { data } = this.props;
        if (data && !isObjectEqual(data, prevProps.data)) {
            this.setLinesListData();
        }
    }

    setLinesListData() {
        const outgoing = this.props.data?.outgoing || {};
        const linesListData: LinesMenuType = {} as LinesMenuType;

        Object.entries(outgoing)?.map(group => {
            const groupName = group[0] as _CHATS;
            const lines = group[1] as {};
            const { items, totalItems } = this.getItems(lines);

            items.sort((l1, l2) => {
                return this.isFavItem(groupName, l2.name)
                    ? 1 : this.isFavItem(groupName, l1.name)
                        ? -1 : l2.chats_count - l1.chats_count;
            });

            BASIC_MENU_TYPES.includes(groupName)
                ? linesListData[groupName] = {
                    items,
                    info: { totalItems },
                }
                : null;
        });

        this.setState({ linesListData }, () => this.checkNotify());
    }

    getItems(lines) {
        const tags = this.props.data?.tags || [];
        let totalItems = 0;

        const items = lines && Object.entries(lines).map(line => {
            const itemCount = Array.isArray(line[1]) ? line[1]?.length : 0;
            totalItems += itemCount;

            return {
                name: line[0],
                display_name: tags?.[line[0]]?.display_name,
                chats_count: itemCount,
            };
        });

        return { items, totalItems };
    }

    isFavItem(menu, name) {
        return this.favLines.match(menu, name);
    }

    checkNotify() {
        const checkTasks = this.favLines.checkTasksCount(this.state.linesListData);
        if (checkTasks) {
            checkTasks.forEach(item => {
                const line = item?.display_name?.toUpperCase() || item.name;
                this.chatNotification.notify(line, FAV_LINES_TYPES.dispatcher);
            });
        }
    }

    openFavouritesConfirm(openConfirm: boolean, favItem, favGroup) {
        this.setState({ favGroup });

        if (!openConfirm) {
            this.favItem = null;
        } else {
            this.favItem = favItem;
        }

        this.setLinesListData();
    }

    getDeferSubmenus() {
        const items = this.getItems(this.props.additionalMenuData?.[_CHATS.DEFFER]);

        return [_CHATS.DEFFER, items];
    }

    render() {
        const {
            selectedItems, onSelect, getArchive, getData, tasksIsLoading, getDeferred,
        } = this.props;
        const { linesListData, favGroup } = this.state;

        const additionalMenus = [
            {
                type: _CHATS.ARCHIVE,
                title: ARCHIVE_TITLE,
                onClick: getArchive.bind(this),
            },
            {
                type: _CHATS.DEFFER,
                title: DEFER_TITLE,
                onClick: getDeferred.bind(this),
                showSubmenu: true,
                submenuData: this.getDeferSubmenus(),
            },
        ];

        return <>
            <LinesList linesListData={linesListData}
                       selectedItems={selectedItems}
                       onSelectLine={onSelect.bind(this)}
                       openFavouritesConfirm={this.openFavouritesConfirm.bind(this, true)}
                       additionalMenus={additionalMenus}
                       additionalControls={[{
                           type: ADDITIONAL_CONTROL_TYPES.UPDATE_BUTTON,
                           onClick: getData.bind(this),
                           state: {
                               isLoading: tasksIsLoading,
                           },
                       }]}/>

            {favGroup
                ? <FavouriteLinesConfirm openAddConfirm={this.openFavouritesConfirm.bind(this)}
                                         group={favGroup}
                                         item={this.favItem}/>
                : null
            }
        </>;
    }
}
