import React, { useContext } from 'react';

import { GetClassification, IGetClassificationExport } from '../../../decorators/GetClassification';
import { ClassificationType } from '../../../types';
import { Button, ButtonTypes } from '../../ui/Button';
import { ShortError } from '../../ui/ErrorBar';
import { Confirm } from '../../ui/FullModal';
import { NoInformation } from '../../ui/NoInformation';
import { isObjectEqual } from '../../utils/isObjectEqual';
import { Request2 } from '../../utils/request';
import ClassificationDialog from '../Chats/ClassificationDialog';
import ShortSessionInfoBlock from '../Chats/ShortSessionInfoBlock';
import { ChatInfoCommon } from '../ChatsCommon/ChatInfoCommon';
import DeferChatUntil from '../ChatsCommon/DeferChatUntil';
import { formatMigrateOptions } from '../ChatsCommon/helpers';
import { MigrateButton } from '../ChatsCommon/MigrateButton';
import { _CHATS } from '../ChatsCommon/types';
import { DAMAGES_PHOTOS_TAGS } from './ChatWindow/constants';
import { ChatsOutgoingContext } from './context';
import * as style from './index.css';
import { requestConfigs, REQUESTS } from './request';

interface InfoProps extends IGetClassificationExport {
    originator_id: string | null;
    user_id: string | null;
    tag_id: string | null;
    chat_id: string | null;
    updateList: (menu?: _CHATS) => void;
    clear: () => void;
    menu: string | null;
    selectedChatItem: any;
    blockRules: any;
    evolutions: any;
    skipClassificationCheck: boolean | undefined;
    hideLastSession: boolean;
}

interface InfoState {
    isWorking: boolean;
    errorControl: Error | null;
    showClassification: boolean;
    closeCallback: any;
    isUndeferWorking: boolean;
    isChatPerformedByUser: boolean;

    migrateError: Error | null;
    migrateOptions: any[];
    confirmIsOpen: boolean;
}

@GetClassification()
export class Info extends React.Component<InfoProps, InfoState> {
    state = {
        isWorking: false,
        errorControl: null,
        showClassification: false,
        closeCallback: () => {},
        isUndeferWorking: false,
        isChatPerformedByUser: false,
        migrateError: null,
        migrateOptions: [],
        confirmIsOpen: false,
    };
    request = new Request2({ requestConfigs });

    shouldComponentUpdate(nextProps: Readonly<InfoProps>, nextState: Readonly<{}>, nextContext: any): boolean {
        return !isObjectEqual(nextProps, this.props) || !isObjectEqual(this.state, nextState);
    }

    componentDidUpdate(prevProps: Readonly<InfoProps>, prevState: Readonly<InfoState>, snapshot?: any) {
        const selectedChat = this.props.selectedChatItem || {};

        if (selectedChat.originator !== prevProps?.selectedChatItem?.originator) {
            this.getEvolutionOptions();
            this.setState({
                isChatPerformedByUser: false,
                errorControl: null,
            });
        } else if (selectedChat.id !== prevProps?.selectedChatItem?.id) {
            this.setState({
                errorControl: null,
            });
        }

        if (this.props.tag_id && this.props.tag_id !== prevProps.tag_id) {
            this.props.getClassification?.(this.props.tag_id);
        }
    }

    perform() {
        this.props.tag_id && this.setState({
            isWorking: true,
            errorControl: null,
        }, () => {
            this.request.exec(REQUESTS.SET_PERFORM, {
                body: {
                    tag_id: this.props.tag_id,
                },
            })
                .then(() => {
                    this.setState({
                        isWorking: false,
                        errorControl: null,
                        isChatPerformedByUser: true,
                    }, () => {
                        this.props.updateList(_CHATS.MY);
                    });
                })
                .catch(errorControl => {
                    this.setState({
                        isWorking: false,
                        errorControl,
                    }, () => {
                        this.props.updateList();
                    });
                });
        });
    }

    dropPerform() {
        this.props.tag_id && this.setState({
            isWorking: true,
            errorControl: null,
        }, () => {
            this.request.exec(REQUESTS.DROP_PERFORM, {
                body: {
                    drop_tag_ids: this.props.tag_id,
                },
            })
                .then(() => {
                    this.setState({
                        isWorking: false,
                        errorControl: null,
                    }, () => {
                        this.props.updateList();
                        this.props.clear();
                    });
                })
                .catch(errorControl => {
                    this.setState({
                        isWorking: false,
                        errorControl,
                    }, () => {
                        this.props.updateList();
                    });
                });
        });
    }

    showClassification(showClassification) {
        this.setState({
            showClassification,
        });
    }

    closeTask() {
        this.setState({
            errorControl: null,
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.DROP_PERFORM_NEW, {
                queryParams: {
                    tag_id: this.props.tag_id,
                    is_finishing: true,
                },
            })
                .then(() => {
                    this.setState({
                        errorControl: null,
                        isWorking: false,
                    }, () => {
                        this.props.updateList();
                        this.props.clear();
                    });
                })
                .catch(errorControl => {
                    this.setState({
                        errorControl,
                        isWorking: false,
                    });
                });
        });
    }

    onDeferred() {
        this.props.updateList();
        this.props.clear();
    }

    beforeClose(closeCallback) {
        if (!this.props.skipClassificationCheck && !this.props.classificationData.length) {
            this.setState({
                closeCallback,
            }, () => {
                this.showClassification(true);
            });
        } else {
            this.setState({
                closeCallback,
                confirmIsOpen: true,
            });
        }
    }

    closeConfirm() {
        this.setState({ confirmIsOpen: false });
    }

    acceptConfirm() {
        this.closeConfirm();
        this.state.closeCallback();
    }

    onClassified() {
        this.showClassification(false);
        this.state.closeCallback();
    }

    undeferChat() {
        const tag_id = this.props.tag_id;
        const chat_id = this.props.selectedChatItem?.topic_link;

        tag_id && chat_id
        && this.setState({ isUndeferWorking: true }, () => {
            this.request.exec(REQUESTS.UNDEFER, {
                queryParams: { tag_id, chat_id },
            })
                .then(() => {
                    this.props.clear();
                    this.props.updateList();
                    this.setState({ isUndeferWorking: false });
                })
                .catch(() => {
                    this.setState({ isUndeferWorking: false });
                });
        });
    }

    getEvolutionOptions() {
        const currentTag = this.props.selectedChatItem?.tag_data?.tag;
        const tagEvolutions = this.props.evolutions?.[currentTag] ?? [];

        if (tagEvolutions.length) {
            this.request.exec(REQUESTS.TAG_DESCRIPTIONS)
                .then(response => {
                    const migrateOptions = formatMigrateOptions(response, tagEvolutions);
                    this.setState({ migrateError: null, migrateOptions });
                })
                .catch(migrateError => {
                    this.setState({ migrateError });
                });
        }
    }

    render() {
        const performer = this.props?.selectedChatItem?.tag_data?.performer;
        const isChatPerformedByUser = this.props.user_id === performer || this.state.isChatPerformedByUser;
        const isOutgoingChatRule = this.props.blockRules.chatAdmin;
        const userId = this.props.originator_id ?? '';
        const tag = this.props?.selectedChatItem?.tag_data?.tag;
        const isDamageTask = DAMAGES_PHOTOS_TAGS.includes(tag);
        const isDeferredChat = this.props.selectedChatItem?.container_tag_data || this.props.menu === _CHATS.DEFFER;

        return !userId
            ? <NoInformation/>
            : <>
                <ChatInfoCommon userId={userId}
                                selectedChatItem={this.props.selectedChatItem}
                                hideLastSession={this.props.hideLastSession}/>

                <div className={style.info_controls}>
                    {
                        this.state.errorControl || this.state.migrateError
                            ? <ShortError error={this.state.errorControl || this.state.migrateError}/>
                            : null
                    }
                    {
                        this.state.showClassification
                        && <ClassificationDialog onClassified={this.onClassified.bind(this)}
                                                 type={ClassificationType.call}
                                                 tag_id={this.props.tag_id}
                                                 user_id={userId}
                                                 chat_id={this.props.chat_id}
                                                 suggest={false}
                                                 keyword={this.props?.selectedChatItem?.tag_data?.tag}
                                                 onClose={this.showClassification.bind(this, false)}/>
                    }
                    {
                        this.props.menu !== _CHATS.ARCHIVE && this.props.menu !== _CHATS.DEFFER
                            ? <>
                                {!isChatPerformedByUser || isOutgoingChatRule
                                    ? <Button colorType={ButtonTypes.positive}
                                              disabled={!this.props.tag_id}
                                              isLoading={this.state.isWorking}
                                              onClick={this.perform.bind(this)}>Взять таску в работу</Button>
                                    : null
                                }

                                {isChatPerformedByUser || isOutgoingChatRule
                                    ? <>
                                        <Button colorType={ButtonTypes.negative}
                                                disabled={!this.props.tag_id}
                                                isLoading={this.state.isWorking}
                                                onClick={this.dropPerform.bind(this)}>Отказаться от задачи</Button>
                                        <Button colorType={ButtonTypes.warning}
                                                disabled={!this.props.tag_id}
                                                isLoading={this.state.isWorking}
                                                onClick={this.beforeClose.bind(this, this.closeTask.bind(this))}>
                                            Завершить
                                        </Button>
                                        <DeferChatUntil onDeferred={this.onDeferred.bind(this)}
                                                        selectedChatItem={this.props.selectedChatItem}/>
                                    </>
                                    : null
                                }

                                {this.state.migrateOptions?.length
                                    ? <MigrateButton options={this.state.migrateOptions}
                                                     chatItem={this.props.selectedChatItem}/>
                                    : null
                                }

                                {
                                    isDamageTask && <>
                                        <DamagesControls isChatPerformedByUser={isChatPerformedByUser}/>
                                    </>
                                }
                            </>
                            : null
                    }
                    {isDeferredChat
                        ? <Button disabled={!this.props.tag_id}
                                  isLoading={this.state.isUndeferWorking}
                                  onClick={this.undeferChat.bind(this)}>Воскресить ✝</Button>
                        : null
                    }
                </div>

                {this.state.confirmIsOpen
                    ? <Confirm onClose={this.closeConfirm.bind(this)}
                               error={null}
                               question={'Закрыть таску?'}
                               accept={this.acceptConfirm.bind(this)}/>
                    : null
                }

            </>;
    }
}

const DamagesControls = (props: { isChatPerformedByUser: boolean }) => {
    const context: any = useContext(ChatsOutgoingContext);
    const length = context?.selectedDamagePhotos?.length;

    return <div>
        {props.isChatPerformedByUser
            ? <Button colorType={ButtonTypes.positive}
                      disabled={!length}
                      onClick={context.showPhotoMarkUpWindow.bind(null, true)}>Разметить фото ({length})
            </Button>
            : null
        }

        <ShortSessionInfoBlock sessionInfo={context?.damageSession}/>
    </div>;
};
