import moment from 'moment';
import React from 'react';
import { RouteProps } from 'react-router-dom';

import { Dict, LSSettingItems } from '../../../types';
import { ONE_SECOND, TIMERS } from '../../constants';
import { SessionHistoryInfoHandler } from '../../models/session';
import { Input } from '../../ui/Input';
import { hasWebphonePanel } from '../../utils/hasWebphonePanel';
import { isObjectEqual } from '../../utils/isObjectEqual';
import LS from '../../utils/localStorage/localStorage';
import { Request2 } from '../../utils/request';
import { deepCopy } from '../../utils/utils';
import { ChatToggle } from '../Chats/ChatToggle';
import { cgiParser, getHrefTagId, getTagId } from '../ChatsCommon/chatUtils';
import * as chatsCommonStyle from '../ChatsCommon/index.css';
import { _CHATS, ISelectedMenus } from '../ChatsCommon/types';
import { ChatWindow } from './ChatWindow/component';
import { MarkupPhoto } from './ChatWindow/DamagesPhotos/MarkupPhoto';
import { ChatsOutgoingContext } from './context';
import DispatcherLinesFetcher from './DispatcherLinesFetcher';
import { Feed, ISelectedChatItem } from './Feed';
import { Info } from './Info';
import { requestConfigs, REQUESTS } from './request';

const OUTGOING_ARCHIVE = '@user_outgoing_communication';
const OUTGOING_TAG_TYPE = 'user_outgoing_communication';

export interface IToggle {
    isOpen: boolean;
}

interface IChatsOutgoingProps extends RouteProps {
    userId?: string;
    blockRules: any;
    evolutions: any;
    dispatcherSettings: any;
    dispatcherLongtermTags: string[];
}

enum TOGGLE_COLUMN {
    LINES = 'lineIsOpen',
    FEED = 'feedIsOpen'
}

export interface ITasks {
    tasks: any;
    menu: string | null;
    subMenu: string | null;
}

export const FINISHED_TASKS = 'user_out_comm_finished';

interface IChatsOutgoingState extends ITasks {
    tasksIsLoading: boolean;
    error: Error | null;

    [TOGGLE_COLUMN.LINES]: boolean;
    [TOGGLE_COLUMN.FEED]: boolean;

    selectedChatItem: ISelectedChatItem | null;
    feedIsLoading: boolean;

    filter: string;
    selectedDamagePhotos: any[];
    showPhotoMarkUpWindow: boolean;
    markupIsLoading: boolean;
    markupError: null | Error;
    allDamagePhotos: any[];
    damageSession: Dict<any>;
    damageDataIsLoading: boolean;
    damageDataError: null | Error;
}

export class ChatsOutgoing extends React.Component<IChatsOutgoingProps, IChatsOutgoingState> {
    request = new Request2({ requestConfigs });
    ls = new LS();
    timeout;
    hasDeferred = false;

    state: IChatsOutgoingState = {
        tasksIsLoading: true,
        error: null,
        tasks: {},
        [TOGGLE_COLUMN.LINES]: this.ls.get(LSSettingItems.chatSettings)?.[TOGGLE_COLUMN.LINES],
        [TOGGLE_COLUMN.FEED]: this.ls.get(LSSettingItems.chatSettings)?.[TOGGLE_COLUMN.FEED],

        menu: null,
        subMenu: null,

        selectedChatItem: null,
        feedIsLoading: false,

        filter: '',

        selectedDamagePhotos: [],
        showPhotoMarkUpWindow: false,
        markupIsLoading: false,
        markupError: null,
        allDamagePhotos: [],
        damageSession: {},
        damageDataIsLoading: false,
        damageDataError: null,
    };

    toggle(field) {
        this.setState((prev) => ({
            ...prev,
            [field]: !prev[field],
        }), () => {
            this.ls.set(LSSettingItems.chatSettings, {
                [TOGGLE_COLUMN.LINES]: this.state[TOGGLE_COLUMN.LINES]
                , [TOGGLE_COLUMN.FEED]: this.state[TOGGLE_COLUMN.FEED],
            });
        });
    }

    shouldComponentUpdate(nextProps: Readonly<any>, nextState: Readonly<any>, nextContext: any): boolean {
        return !isObjectEqual(nextProps, this.props) || !isObjectEqual(nextState, this.state);
    }

    componentDidMount(): void {
        this.getData();
        this.checkCgi();
    }

    componentDidUpdate(
        prevProps: Readonly<IChatsOutgoingProps>,
        prevState: Readonly<IChatsOutgoingState>,
        snapshot?: any,
    ): void {
        if (this.props?.location?.search !== prevProps.location?.search) {
            this.checkCgi();
        }
    }

    componentWillUnmount(): void {
        clearTimeout(this.timeout);
    }

    checkCgi() {
        const { location } = this.props;
        const [user_id, chat_id] = cgiParser(location);

        if (user_id && chat_id) {
            const monthCount = 3;
            const since = Math.round(+moment().subtract(monthCount, 'month') / ONE_SECOND);
            const until = Math.round(+moment() / ONE_SECOND);

            this.request.exec(REQUESTS.LIST, {
                body: { chats: [{ user_id, chat_id }] },
            }).then(response => {
                const selectedChatItem = response?.chats?.[0] && {
                    ...response.chats[0],
                    topic_link: response.chats[0].id,
                    id: response.chats[0].tag_id,
                } || null;

                if (!selectedChatItem.tag_data) {
                    const tag_id = getTagId(selectedChatItem) || getHrefTagId(this.props.location);

                    tag_id && this.request.exec(REQUESTS.GET_TAG_DETAILS, {
                        queryParams: { tag_id, since, until },
                    })
                        .then(response => {
                            const tag_data = response.records?.[0]?.tag_details;
                            tag_data ? selectedChatItem.tag_data = tag_data : null;
                        });
                }

                this.setState({
                    selectedChatItem,
                    menu: null,
                    subMenu: null,
                });
            });
        }
    }

    buildOutgoingObj(outgoingData, filter?: any) {
        const outgoing = (outgoingData || []).filter(filter || this.commonFilter);

        return outgoing.reduce((_p, _c) => {
            const tag_data = _c?.tag_data || _c?.tag_details || {};
            _c.last_message = { text: tag_data.comment, type: 'plaintext', timestamp: _c.timestamp };
            _c.originator = tag_data.object_id || _c.object_id;
            _c.id = tag_data.tag_id || _c.tag_id;
            _c.topic_link = tag_data.topic_link || _c.topic_link;
            _c.tag_data = tag_data;

            const tag = tag_data.tag || _c.tag;

            if (!_p.hasOwnProperty(tag)) {
                _p[tag] = [_c];
            } else {
                _p[tag].push(_c);
            }

            return _p;
        }, {}) || {};
    }

    commonFilter(el) {
        return el?.tag_data?.tag !== FINISHED_TASKS;
    }

    getData() {
        clearTimeout(this.timeout);
        this.setState({
            tasksIsLoading: true,
        }, () => {
            Promise.all([
                this.request.exec(REQUESTS.GET_TASKS, { queryParams: { performer_id: this.props.userId } }), //my
                this.request.exec(REQUESTS.GET_TASKS, { queryParams: { performer_id: '0' } }), //new
                this.request.exec(REQUESTS.GET_TASKS), //all
            ])
                .then((response) => {
                    const { tasks } = this.state;
                    const ALL_TASKS_RESPONSE = 2;
                    const outgoing = {
                        [_CHATS.MY]: this.buildOutgoingObj(response[0].outgoing),
                        [_CHATS.NEW]: this.buildOutgoingObj(response[1].outgoing),
                        [_CHATS.ALL]: this.buildOutgoingObj(response[ALL_TASKS_RESPONSE].outgoing),
                        [_CHATS.ARCHIVE]: tasks?.outgoing?.[_CHATS.ARCHIVE] || {},
                        [_CHATS.DEFFER]: tasks?.outgoing?.[_CHATS.DEFFER] || {},
                    };

                    let users = [...response[0].users, ...response[1].users, ...response[ALL_TASKS_RESPONSE].users]
                        .reduce((_p, _c) => {
                            if (!_p.hasOwnProperty(_c.id)) {
                                _p[_c.id] = _c;
                            }

                            return _p;
                        }, {});

                    users = { ...users, ...tasks.users };

                    const tags = [...response[0].tags, ...response[1].tags, ...response[ALL_TASKS_RESPONSE].tags]
                        .reduce((_p, _c) => {
                            if (!_p.hasOwnProperty(_c.name)) {
                                _p[_c.name] = _c;
                            }

                            return _p;
                        }, {});

                    this.setState({
                        tasksIsLoading: false,
                        error: null,
                        tasks: {
                            tags,
                            users,
                            outgoing,
                        },
                    }, () => {
                        const deferredItems = Object.keys(this.state.tasks.outgoing?.[_CHATS.DEFFER]);

                        if (!deferredItems?.length && !this.hasDeferred) {
                            this.hasDeferred = true;
                            this.getDeferred();
                        }
                    });

                    this.repeat();
                })
                .catch((error) => {
                    this.setState({
                        tasksIsLoading: false,
                        error,
                    });
                });
        });
    }

    repeat() {
        this.timeout = setTimeout(this.getData.bind(this), TIMERS.OUTGOING_CHATS);
    }

    setMenu(menu, subMenu) {
        location.href = '#/outgoing';
        this.setState({
            menu,
            subMenu,
            selectedChatItem: null,
        });
    }

    getArchive() {
        this.setMenu(_CHATS.ARCHIVE, null);
        this.setState({
            feedIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.ARCHIVE, { queryParams: { tags: OUTGOING_ARCHIVE, rev: true } })
                .then(response => {
                    let tasks = deepCopy(this.state.tasks);
                    tasks.outgoing[_CHATS.ARCHIVE] = this.buildOutgoingObj(response.tags);
                    tasks = this.addTaskUsers(response, tasks);

                    this.setState({
                        feedIsLoading: false,
                        tasks,
                    });
                });
        });
    }

    getDeferred(menu?: string, submenu?: string) {
        menu && this.setMenu(_CHATS.DEFFER, submenu);

        !submenu && this.setState({ feedIsLoading: true }, () => {
            this.request.exec(REQUESTS.GET_DEFERRED, {
                queryParams: {
                    contained_tag_types: OUTGOING_TAG_TYPE,
                },
            })
                .then(response => {
                    const deferredChats = response.container?.map(el => {
                        return { tag_data: el };
                    });
                    let tasks = deepCopy(this.state.tasks);

                    tasks.outgoing && (tasks.outgoing[_CHATS.DEFFER] = this.buildOutgoingObj(deferredChats));
                    tasks = this.addTaskUsers(response, tasks);

                    this.setState({
                        feedIsLoading: false,
                        tasks,
                    });
                })
                .catch(() => {
                    const tasks = deepCopy(this.state.tasks);
                    tasks.outgoing && (tasks.outgoing[_CHATS.DEFFER] = {});
                    this.setState({ feedIsLoading: false, tasks });
                });
        });
    }

    addTaskUsers(response, tasks) {
        response.users?.forEach(user => {
            if (tasks.users && !tasks.users[user.id]) {
                tasks.users[user.id] = user;
            }
        });

        return tasks;
    }

    clear() {
        this.setState({
            selectedChatItem: null,
        });
    }

    selectChatItem(chat) {
        this.setState({
            selectedChatItem: {
                ...chat,
                id: chat.id,
                originator: chat.originator,
                topic_link: chat.topic_link,
            },
        });
    }

    updateList(menu?: _CHATS) {
        menu
            ? this.setState({ menu }, () => this.getData())
            : this.getData();
    }

    onChange(filter) {
        this.setState({ filter });
    }

    selectDamagePhoto(item) {
        const items = [...this.state.selectedDamagePhotos];
        const position = items.indexOf(item);

        if (position > -1) {
            items.splice(position, 1);
            this.setState({
                selectedDamagePhotos: items,
            });
        } else {
            this.setState({
                selectedDamagePhotos: [...items, item],
            });
        }
    }

    clearDamagesPhotos() {
        this.setState({
            selectedDamagePhotos: [],
        });
    }

    showPhotoMarkUpWindow(showPhotoMarkUpWindow) {
        this.setState({
            showPhotoMarkUpWindow,
        });
    }

    markup(value) {
        this.setState({
            markupIsLoading: true,
        }, () => {
            const queue = this.state.selectedDamagePhotos?.map(el => {
                let meta_data = el.meta_data;

                if (!meta_data || !meta_data.hasOwnProperty('support_verdicts')) {
                    meta_data = { support_verdicts: [{}] };
                }

                if (value === null) { // remove support_verdicts by btn
                    meta_data.support_verdicts = [{}];
                } else {
                    meta_data.support_verdicts = value.support_verdicts;
                }

                meta_data.support_verdicts_timestamp = new Date().getTime();

                return this.request.exec(REQUESTS.MARK_UP, {
                    body: {
                        image_id: el.image_id,
                        meta_data,
                    },
                });
            });

            Promise.all(queue)
                .then(() => {
                    this.setState({
                        markupIsLoading: false,
                        markupError: null,
                        showPhotoMarkUpWindow: false,
                    }, () => {
                        const session_id = SessionHistoryInfoHandler.getSessionId.call(this.state.damageSession);
                        session_id && this.getDamageData(session_id);
                    });
                })
                .catch(markupError => {
                    this.setState({
                        markupIsLoading: false,
                        markupError,
                    });
                });

        });
    }

    getDamageData(session_id) {
        this.setState({
            damageDataIsLoading: true,
            selectedDamagePhotos: [],
            allDamagePhotos: [],
        }, () => {
            this.request.exec(REQUESTS.GET_SESSION, {
                queryParams: {
                    session_id: session_id,
                },
            }).then(response => {
                const car_id = SessionHistoryInfoHandler.getCarId.call(response);
                const since = SessionHistoryInfoHandler.getStart.call(response) / ONE_SECOND;
                const until = SessionHistoryInfoHandler.getFinish.call(response) / ONE_SECOND;
                this.setState({
                    damageSession: response,
                }, () => {

                    car_id && session_id && this.request.exec(REQUESTS.GET_DAMAGES_PHOTOS, {
                        queryParams: {
                            session_id,
                            car_id,
                            since,
                            until,
                        },
                    }).then(response => {
                        this.setState({
                            damageDataError: null,
                            allDamagePhotos: response?.images || [],
                            damageDataIsLoading: false,
                        });
                    }).catch(damageDataError => {
                        this.setState({
                            damageDataError,
                            damageDataIsLoading: false,
                        });
                    });
                });

            }).catch(damageDataError => {
                this.setState({
                    damageDataError,
                    damageDataIsLoading: false,
                });
            });
        });
    }

    render() {
        const compileStyle = `${chatsCommonStyle.chats}`
            + ` ${this.state.lineIsOpen ? `${chatsCommonStyle.line_is_open}` : ''}`
            + ` ${hasWebphonePanel(location.href, this.props.blockRules) ? `${chatsCommonStyle.webphone_is_open}` : ''}`
            + ` ${this.state.feedIsOpen ? `${chatsCommonStyle.feed_is_open}` : ''}`;

        const selectedItems: ISelectedMenus = {
            menuItem: this.state.menu,
            subMenuItem: this.state.subMenu,
        };

        const additionalMenuData = {
            [_CHATS.DEFFER]: this.state.tasks.outgoing?.[_CHATS.DEFFER] || {},
        };

        const tag = this.state.subMenu ?? this.state.selectedChatItem?.tag_data?.tag;
        const currentQueueSetting = this.props.dispatcherSettings?.find(el => el.queue === tag) ?? {};
        const isEditorClosedByDefault = currentQueueSetting.close_text_editor;
        const skipClassificationCheck = currentQueueSetting.skip_classification_check;
        const hideLastSession = currentQueueSetting.hide_last_session;

        return <div className={compileStyle}>
            <ChatsOutgoingContext.Provider value={{
                selectDamagePhoto: this.selectDamagePhoto.bind(this),
                showPhotoMarkUpWindow: this.showPhotoMarkUpWindow.bind(this),
                selectedDamagePhotos: this.state.selectedDamagePhotos,
                getDamageData: this.getDamageData.bind(this),
                allDamagePhotos: this.state.allDamagePhotos,
                damageSession: this.state.damageSession,
                damageDataIsLoading: this.state.damageDataIsLoading,
                damageDataError: this.state.damageDataError,
            }}>
                <div className={`${chatsCommonStyle.chats_column} ${chatsCommonStyle.chat_lines}`}>
                    <ChatToggle isOpen={this.state[TOGGLE_COLUMN.LINES]}
                                toggle={this.toggle.bind(this, TOGGLE_COLUMN.LINES)}/>

                    {this.state[TOGGLE_COLUMN.LINES]
                        ? <DispatcherLinesFetcher data={this.state.tasks}
                                                  getData={this.getData.bind(this)}
                                                  tasksIsLoading={this.state.tasksIsLoading}
                                                  selectedItems={selectedItems}
                                                  getArchive={this.getArchive.bind(this)}
                                                  getDeferred={this.getDeferred.bind(this)}
                                                  additionalMenuData={additionalMenuData}
                                                  onSelect={this.setMenu.bind(this)}/>
                        : null
                    }
                </div>

                <div className={`${chatsCommonStyle.chats_column} ${chatsCommonStyle.common_feed} ${chatsCommonStyle.outgoing}`}>
                    <div className={chatsCommonStyle.controls}>
                        <ChatToggle isOpen={this.state[TOGGLE_COLUMN.FEED]}
                                    toggle={this.toggle.bind(this, TOGGLE_COLUMN.FEED)}/>

                        {this.state.feedIsOpen && this.state.menu
                            ? <Input value={this.state.filter}
                                     onChange={this.onChange.bind(this)}
                                     placeholder={'фильтр'}/>
                            : null
                        }
                    </div>
                    <Feed isOpen={this.state.feedIsOpen}
                          filterValue={this.state.filter}
                          menu={this.state.menu}
                          subMenu={this.state.subMenu}
                          selectedChatItem={this.state.selectedChatItem}
                          selectChatItem={this.selectChatItem.bind(this)}
                          feedIsLoading={this.state.feedIsLoading}
                          tasks={this.state.tasks}/>
                </div>
                <div className={`${chatsCommonStyle.chats_column} ${chatsCommonStyle.chat_window}`}>
                    <ChatWindow topic_link={this.state.selectedChatItem?.topic_link || null}
                                user_id={this.state.selectedChatItem?.originator || null}
                                tag_id={this.state.selectedChatItem?.id || null}
                                selectedChatItem={this.state.selectedChatItem || null}
                                tasks={this.state.tasks}
                                menu={this.state.menu}
                                dispatcherLongtermTags={this.props.dispatcherLongtermTags}
                                isEditorClosedByDefault={isEditorClosedByDefault}/>
                </div>
                <div className={`${chatsCommonStyle.chats_column} ${chatsCommonStyle.chat_info}`}>
                    <Info originator_id={this.state.selectedChatItem?.originator || null}
                          user_id={this.props.userId ?? ''}
                          clear={this.clear.bind(this)}
                          updateList={this.updateList.bind(this)}
                          tag_id={this.state.selectedChatItem?.id || null}
                          chat_id={this.state.selectedChatItem?.topic_link || null}
                          selectedChatItem={this.state.selectedChatItem}
                          menu={this.state.menu}
                          blockRules={this.props.blockRules}
                          evolutions={this.props.evolutions}
                          hideLastSession={hideLastSession}
                          skipClassificationCheck={skipClassificationCheck}/>
                </div>
            </ChatsOutgoingContext.Provider>
            {
                this.state.showPhotoMarkUpWindow &&
                <MarkupPhoto onClose={this.showPhotoMarkUpWindow.bind(this, false)}
                             error={this.state.markupError}
                             isLoading={this.state.markupIsLoading}
                             initialData={this.state.selectedDamagePhotos} /*only for one photo*/
                             markup={this.markup.bind(this)}/>
            }
        </div>;
    }
}
