import moment from 'moment';
import React from 'react';

import { ONE_SECOND, ROBOT_FRONTEND, STAFF_ONLY } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import FormatDate from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import * as styleTable from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import Spin from '../../Spin';
import { ACTION_HISTORY_TITLE } from '../constants';
import { CHAT_REQUESTS, REQUESTS } from '../request';
import { ChatHistoryItemType, IChatHistoryItem } from '../types';
import * as style from './index.css';
import { Item } from './utils';

interface IChatHistoryViewProps {
    onClose: () => void;
    user_id: string;
    chat_id: string;
    tag_id: string;
    linked_tag_id?: any;
}

interface IChatHistoryViewState {
    error: Error | null;
    isLoading: boolean;
    history: IChatHistoryItem[];
}

export class ChatHistoryView extends React.Component<IChatHistoryViewProps, IChatHistoryViewState> {
    state: IChatHistoryViewState = {
        error: null,
        isLoading: false,
        history: [] as IChatHistoryItem[],
    };
    request = new Request2({ requestConfigs: CHAT_REQUESTS });

    componentDidMount() {
        this.getHistory();
    }

    getType(message, index) {

        if (index === 0) {
            return ChatHistoryItemType.CREATED_CHAT;
        }

        if (message?.traits?.includes(STAFF_ONLY)) {
            return ChatHistoryItemType.NOTE;
        }

        if (ChatHistoryItemType[message?.type?.toUpperCase()]) {
            return ChatHistoryItemType[message?.type?.toUpperCase()];
        }

        return ChatHistoryItemType.COMMON_MESSAGE;
    }

    getClassificationHistory(tagId?: string) {
        tagId && this.setState(() => ({
            isLoading: true,
        }), () => {
            this.request.exec(REQUESTS.GET_CLASSIFICATION_HISTORY,
                { queryParams: { tag_id: tagId } },
            )
                .then(response => {
                    const history: IChatHistoryItem[] = response?.items?.map((el) => {
                        return {
                            timestamp: el.timestamp,
                            historyAction: el.action,
                            type: ChatHistoryItemType.CLASSIFY,
                            user: el.operator_id,
                            content: el,
                        };
                    });
                    this.setState((prev) => {
                        return {
                            isLoading: false,
                            error: null,
                            history: [...prev.history, ...history],
                        };
                    });
                })
                .catch(error => {
                    this.setState(() => ({
                        error,
                        isLoading: false,
                    }));
                });
        });
    }

    getDeepHistory() {
        const { user_id, chat_id, tag_id } = this.props;
        const firstMessage = this.state.history?.[0]?.timestamp;
        user_id && this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_USER_TAGS_HISTORY, {
                queryParams: {
                    object_id: user_id,
                    since: firstMessage || Math.round(+moment().subtract(1, 'month') / ONE_SECOND),
                    until: Math.round(+moment() / ONE_SECOND),
                },
            })
                .then(response => {
                    const tagId = tag_id || response?.records
                        ?.find((el) => el.tag_details?.topic_link === chat_id)?.tag_id;
                    const currentHistory = this.formatDeepHistory(response, tagId);

                    this.setState((prev) => {
                        return {
                            isLoading: false,
                            error: null,
                            history: [...prev.history, ...currentHistory],
                        };
                    }, () => {
                        this.getClassificationHistory(tagId);
                    });
                })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    }, () => {
                        this.getClassificationHistory();
                    });
                });
        });
    }

    formatDeepHistory(response, tagId): any[] {
        const history: any = [];

        response?.records?.forEach(tag => {
            const isLinked = tag.tag_id === this.props.linked_tag_id;
            const currentTag = tag.tag_id === tagId;

            if (currentTag || isLinked) {
                history.push({
                    timestamp: tag.timestamp,
                    historyAction: tag.action,
                    type: ChatHistoryItemType.DEEP_HISTORY,
                    user: tag.user_data_full ?? tag.user_id,
                    content: tag,
                });
            }
        });

        return history;
    }

    getHistory() {
        this.setState({
            isLoading: true,
        }, () => {

            this.request.exec(REQUESTS.HISTORY, {
                queryParams: {
                    chat_id: this.props.chat_id,
                    user_id: this.props.user_id,
                },
            }).then(response => {

                const users = response?.users?.reduce((_p, _c) => {
                    if (!_p.hasOwnProperty(_c.id)) {
                        _p[_c.id] = _c;
                    }

                    return _p;
                }, {});

                const history: IChatHistoryItem[] = response?.messages?.map((message, index) => {
                    return {
                        timestamp: message.timestamp,
                        type: this.getType(message, index),
                        user: users[message.author] || message.author,
                        isRobot: message.author === ROBOT_FRONTEND,
                        content: message,
                    };
                });

                if (response?.expected_action?.type === ChatHistoryItemType.CLOSED_CHAT) {
                    history.push({
                        timestamp: history[history.length - 1].timestamp,
                        type: ChatHistoryItemType.CLOSED_CHAT,
                        user: '',
                    });
                }

                this.setState({
                    isLoading: false,
                    error: null,
                    history,
                }, () => {
                    this.getDeepHistory();
                });
            })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    });
                });
        });

    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const { error, history, isLoading } = this.state;

        return <Window error={error}
                       className={style.component}
                       onClose={this.props.onClose.bind(this)}
                       title={ACTION_HISTORY_TITLE}>
            {isLoading
                ? <Spin/>
                : null
            }

            {history.length
                ? <table className={`${styleTable.table}`}>
                    <thead>
                        <tr>
                            <th>#</th>
                            <th>дата</th>
                            <th>пользователь</th>
                            <th>действие</th>
                        </tr>
                    </thead>
                    <tbody>
                        {
                            history
                                .sort((a, b) => a.timestamp - b.timestamp)
                                .map((item, index) => {
                                    const isClassification = item.type === 'classify';
                                    let userData = item.user;

                                    if (isClassification) {
                                        userData = this.state.history?.find(el => el.user.id === item.user)?.user;
                                    }

                                    const printName = UserInfoHandler.getPrintName.call(userData);
                                    const userId = UserInfoHandler.getId.call(userData);

                                    return <tr key={index}>
                                        <td>{index + 1}</td>
                                        <td><FormatDate value={item.timestamp * ONE_SECOND} withSecond/></td>
                                        <td>
                                            {printName
                                                ? <Link href={`#/clients/${userId}/info`} target={'_blank'}>
                                                    {printName}
                                                </Link>
                                                : item.user
                                            }
                                        </td>
                                        <td>
                                            <Item item={item}/>
                                        </td>
                                    </tr>;
                                })
                        }
                    </tbody>
                </table>
                : <NoInformation/>
            }
        </Window>;
    }
}
