import moment from 'moment';
import React from 'react';

import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { FormatDateInString } from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import { ChatHistoryItemTypeDescription } from '../constants';
import {
    ChatHistoryItemType,
    CLASSIFICATION_ACTIONS,
    DEEP_HISTORY_ACTION_TYPES,
    EVOLVE_TAGS,
    IChatHistoryItem,
} from '../types';
import * as style from './index.css';

export const Item = React.memo((props: { item }) => {
    const { item } = props;

    switch (item.type) {
    case ChatHistoryItemType.DEEP_HISTORY:
        return <DeepHistoryAction item={item}/>;
    case ChatHistoryItemType.CLASSIFY:
        return <ClassifyHistoryAction item={item}/>;
    case ChatHistoryItemType.NOTE:
        return <HistoryNote item={item} />;
    default:
        return <CommonHistoryAction item={item}/>;
    }
});

const DeepHistoryAction = React.memo((props: { item: IChatHistoryItem }) => {
    const { item } = props;

    function matchEvolveActions() {
        switch (item.content?.tag_name) {
        case EVOLVE_TAGS.CONTAINER_TAG:
            return <div>Отложен {formatContainerString(item.content) ?? EMPTY_DATA}</div>;
        case EVOLVE_TAGS.LONG_TERM_PREPARATION:
            return <div>Подбор {getCarLink(item.content)}</div>;
        default:
            return <div>
                <span className={style.deep_history_action}>{item.historyAction}</span>
                {` — Перемещен в очередь "${item.content?.tag_display_name}"`}
            </div>;
        }
    }

    switch (item.historyAction) {
    case DEEP_HISTORY_ACTION_TYPES.SET_PERFORMER:
        return <div>Взят в работу</div>;
    case DEEP_HISTORY_ACTION_TYPES.DROP_PERFORMER:
        return <div>Работа завершена</div>;
    case DEEP_HISTORY_ACTION_TYPES.EVOLVE:
        return matchEvolveActions();
    case DEEP_HISTORY_ACTION_TYPES.ADD:
        return <div>
            <span className={style.deep_history_action}>{item.historyAction}</span>
            {` — Назначен в очередь "${item.content?.tag_display_name}"`}
        </div>;
    default:
        return <div className={style.deep_history_action}>
            {item.historyAction}
        </div>;
    }
});

const ClassifyHistoryAction = React.memo((props: { item: IChatHistoryItem }) => {
    const { item } = props;

    function translateAction(action) {
        switch (action) {
        case CLASSIFICATION_ACTIONS.ADD:
            return 'Добавлена классификация';
        case CLASSIFICATION_ACTIONS.REMOVE:
            return 'Удалена классификация';
        default:
            return action;
        }
    }

    return <div className={style.classify_history_action}>
        <div>{translateAction(item.historyAction)}</div>
        <div>{item.content?.category?.meta?.label}</div>
    </div>;
});

const HistoryNote = React.memo((props: { item: IChatHistoryItem }) => {
    return <div className={style.history_note}>
        <span className={style.classify_history_action}>Комментарий: </span>
        {props.item?.content?.text ?? EMPTY_DATA}
    </div>;
});

const CommonHistoryAction = React.memo((props: { item: IChatHistoryItem }) => {
    const { item } = props;

    return <div>
        {ChatHistoryItemTypeDescription[item.type]}
    </div>;
});

function formatContainerString(item) {
    const start = moment(item.timestamp * ONE_SECOND);
    const end = moment(item.tag_details.SLAInstant * ONE_SECOND);
    const finalValue = FormatDateInString({
        value: item.tag_details.SLAInstant * ONE_SECOND,
        withoutYear: true,
    });

    const finalString = end.from(start)?.replace('через', 'на');

    return finalString + ` (до ${finalValue})`;
}

function getCarLink(item) {
    const car_id = item.tag_details?.CarId || '';
    const carHref = car_id ? `#/cars/${car_id}/info` : '';

    return <Link href={carHref} target={'_blank'}>машины</Link>;
}
