import React, { useEffect, useState } from 'react';

import { Dict } from '../../../../types';
import { useRequestHandler } from '../../../hooks/useRequestHandler';
import { Button, CancelButton } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import * as coreStyle from '../../../ui/index.css';
import Select, { IOptionInfo } from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import { CHAT_REQUESTS, REQUESTS } from '../request';

interface IJumpToNodeModalProps {
    onClose: () => void;
    chat_id: string;
    user_id: string;
}

export const JumpToNodeModal = React.memo((props: IJumpToNodeModalProps) => {
    const { onClose, chat_id, user_id } = props;
    const request = new Request2({ requestConfigs: CHAT_REQUESTS });

    const [options, setOptions] = useState<IOptionInfo[]>([]);
    const [selectedOption, setSelectedOption] = useState<IOptionInfo | null>(null);

    const indexOfPoint = chat_id.indexOf('.');
    const notifierName = chat_id.slice(0, indexOfPoint === -1 ? chat_id.length : indexOfPoint);

    const notifierRequestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_NOTIFIERS,
            requestOptions: { queryParams: { ids: notifierName } },
        };
    }, []);

    const jumpRequestOptions = React.useMemo(() => {
        return {
            requestName: REQUESTS.JUMP_TO_NODE,
            requestOptions: {
                body: {
                    chat_id,
                    user_id,
                    node_name: selectedOption,
                },
            },
        };
    }, [selectedOption]);

    const [
        isNotifLoading, ,
        notifError,
        notifExec,
    ] = useRequestHandler(request, notifierRequestOptions, createSelectOptions);
    const [isJumpLoading, , jumpError, execJump] = useRequestHandler(request, jumpRequestOptions, onClose);

    useEffect(() => {
        notifierName && notifExec();
    }, []);

    function createSelectOptions(response) {
        const notifier = response?.objects?.[0] ?? {};
        const nodesToJump: Dict<string>[] = notifier.meta?.chat_script?.nodes_to_jump ?? [];
        const options: IOptionInfo[] = nodesToJump.map((node: {value: string; text: string}) => {
            const { value, text } = node;

            return { value, text };
        });

        setOptions(options);
    }

    return <Window onClose={onClose.bind(null)} title={'Перейти в ноду'} error={notifError || jumpError}>
        <Select options={options}
                onSelect={setSelectedOption.bind(null)}
                isLoading={isNotifLoading}
                placeholder={'Нода для перехода'}/>

        <div className={coreStyle.button_container}>
            <CancelButton onClick={onClose.bind(null)}/>

            <Button onClick={execJump.bind(null)}
                    isLoading={isJumpLoading}
                    disabled={!selectedOption}>
                Перейти
            </Button>
        </div>
    </Window>;
});
