import React from 'react';

import { GetClassification, IGetClassificationExport } from '../../../../decorators/GetClassification';
import { ClassificationType } from '../../../../types';
import { Button, ButtonTypes } from '../../../ui/Button';
import { ShortError } from '../../../ui/ErrorBar';
import { Confirm } from '../../../ui/FullModal';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { combineRequests, Request2 } from '../../../utils/request';
import { REQUESTS } from '../../ChatOutgoing/request';
import { ChatInfoCommon } from '../../ChatsCommon/ChatInfoCommon';
import DeferChatUntil from '../../ChatsCommon/DeferChatUntil';
import { formatMigrateOptions } from '../../ChatsCommon/helpers';
import { MigrateButton } from '../../ChatsCommon/MigrateButton';
import { CLIENTS_CARD_REQUESTS } from '../../Clients/request';
import ClassificationDialog from '../ClassificationDialog';
import { DEFERRED_MARK } from '../constants';
import { CHAT_REQUESTS, REQUESTS as CH_REQUEST } from '../request';
import { AddComment } from './AddComment';
import { ChatInfoMapDispatchToProps, IMapStateToProps } from './index';
import * as style from './index.css';
import { JumpToNodeModal } from './JumpToNodeModal';

interface IChatInfoProps extends IMapStateToProps, ChatInfoMapDispatchToProps, IGetClassificationExport {
}

const CLOSE_CHAT_ACTIONS = {
    close: 'close',
    close_light: 'close_light',
    close_ignore_external: 'close_ignore_external',
};

@GetClassification()
export class ChatInfo extends React.Component<IChatInfoProps, any> {
    state = {
        isEmpty: true,
        isWorking: false,
        errorControl: null,
        isLoading: false,
        error: null,

        confirmIsOpen: false,
        acceptAction: () => {},
        question: '',

        showCommentModal: false,

        showClassificationDialog: false,
        closeCallback: () => {},
        showChatHistory: false,

        isRecoverLoading: false,
        recoverError: null,
        showRecoverConfirm: false,
        isChatPerformedByUser: false,

        migrateError: null,
        migrateOptions: [],

        isJumpModalOpen: false,

        recovered: false,
    };
    request = new Request2({ requestConfigs: combineRequests(CLIENTS_CARD_REQUESTS, CHAT_REQUESTS) });

    componentDidUpdate(prevProps: Readonly<any>, prevState: Readonly<any>, snapshot?: any): void {
        const chat_id = this.props.selectedChatItem?.id ?? '';
        const prev_chat_id = prevProps.selectedChatItem?.id ?? '';
        const originatorPrev = this.getId(prevProps);
        const originatorNew = this.getId(this.props);

        if (originatorPrev !== originatorNew) {
            this.request.abort();
            this.getEvolutionOptions();

            this.setState({
                errorControl: null,
            });
        } else if (chat_id !== prev_chat_id) {
            this.setState({
                errorControl: null,
                recovered: false,
            });
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getId(props) {
        return props.selectedChatItem?.originator;
    }

    getTagId() {
        return this.props.selectedChatItem?.tag_data?.tag_id
            || this.props.selectedChatItem?.tag_id;
    }

    dropPerform() {
        this.setState({
            isWorking: true,
            errorControl: null,
        }, () => {
            this.request.exec(CH_REQUEST.DROP_PERFORM, {
                body: {
                    drop_tag_ids: this.getTagId(),
                },
            })
                .then(() => {
                    this.updateChatPerformer(null);

                    this.setState({
                        isWorking: false,
                    }, () => {
                        this.props?.cancelChatCallBack();
                    });
                })
                .catch(errorControl => {
                    this.setState({
                        isWorking: false,
                        errorControl,
                    });
                });
        });

    }

    perform() {
        const tag_id = this.getTagId();

        tag_id && this.setState({
            isWorking: true,
            errorControl: null,
        }, () => {
            this.request.exec(CH_REQUEST.SET_PERFORM, {
                body: { tag_id },
            })
                .then(() => {
                    this.updateChatPerformer(this.props.userId);

                    this.setState({
                        isWorking: false,
                        errorControl: null,
                    });
                })
                .catch(errorControl => {
                    this.setState({
                        isWorking: false,
                        errorControl,
                    });
                });
        });

    }

    updateChatPerformer(newPerformer) {
        const tag = Object.assign({}, this.props.selectedChatItem?.tag_data);
        tag.performer = newPerformer;
        this.props.updateChatItemTag(this.props.selectedChatItem, tag);
    }

    showMigrate(state) {
        this.setState({
            showMigrate: state,
        });
    }

    switchConfirm(confirmIsOpen) {
        this.setState({
            confirmIsOpen,
            isWorking: false,
            error: null,
            errorControl: null,
        });
    }

    closeChatWithConfirm(ignore_external?: boolean) {
        this.setState({
            question: <div>Проблема решена? Чатик <b>закрываем!</b>
                <div>Больше в него <strong>никто не напишет</strong>!!! 😿</div>
            </div>,
            confirmIsOpen: true,
            errorControl: null,
            acceptAction: () => {
                this.closeChat(ignore_external);
            },
        });
    }

    closeChat(ignore_external?: boolean) {
        const queryParams = {
            tag_id: this.getTagId(),
            is_finishing: true,
        };

        ignore_external ? queryParams['ignore_external'] = true : null;

        this.setState({
            isWorking: true,
        }, () => {
            this.request.exec(CH_REQUEST.DROP_PERFORM_NEW, { queryParams })
                .then(() => {
                    this.switchConfirm(false);
                    this.props?.cancelChatCallBack();
                })
                .catch(errorControl => {
                    this.setState({
                        errorControl,
                        isWorking: false,
                    });
                });
        });
    }

    closeChatLight() {
        this.setState({
            isWorking: true,
            errorControl: null,
        }, () => {
            this.request.exec(CH_REQUEST.DROP_PERFORM_NEW, {
                queryParams: {
                    tag_id: this.getTagId(),
                },
            })
                .then(() => {
                    this.switchConfirm(false);
                    this.props?.cancelChatCallBack();
                })
                .catch(errorControl => {
                    this.setState({
                        errorControl,
                        isWorking: false,
                    });
                });
        });
    }

    showCommentModal(state) {
        this.setState({
            showCommentModal: state,
        });
    }

    showClassificationDialog(showClassificationDialog) {
        this.setState({
            showClassificationDialog,
        });
    }

    onClassified() {
        this.showClassificationDialog(false);
        this.state.closeCallback();
    }

    async beforeClose(action) {
        const isCloseAction = action === CLOSE_CHAT_ACTIONS.close;
        const isCloseIgnoreExternal = action === CLOSE_CHAT_ACTIONS.close_ignore_external;
        const tag_id = this.getTagId();
        const categorizations = await this.props.getClassificationResponse?.(tag_id);

        if (!categorizations.length) {
            this.setState({
                closeCallback: isCloseAction
                    ? this.closeChat.bind(this)
                    : isCloseIgnoreExternal
                        ? this.closeChat.bind(this, true)
                        : this.closeChatLight.bind(this),
            }, () => {
                this.showClassificationDialog(true);
            });
        } else {
            isCloseAction
                ? this.closeChatWithConfirm()
                : isCloseIgnoreExternal
                    ? this.closeChatWithConfirm(true)
                    : this.closeChatLight();
        }
    }

    shouldComponentUpdate(nextProps: Readonly<IChatInfoProps>, nextState: Readonly<any>, nextContext: any): boolean {
        return !isObjectEqual(this.state, nextState) || !isObjectEqual(this.props, nextProps);
    }

    onDeferred() {
        this.switchConfirm(false);
        this.props?.cancelChatCallBack();
    }

    showChatHistory(showChatHistory) {
        this.setState({
            showChatHistory,
        });
    }

    showRecoverConfirm(showRecoverConfirm) {
        this.setState({ showRecoverConfirm });
    }

    onChatRecover() {
        const tag_id = this.getTagId();
        const chat_id = this.props.selectedChatItem?.id;

        this.setState({ isRecoverLoading: true, recoverError: null, recovered: false }, () => {
            this.request.exec(CH_REQUEST.UNDEFER, { queryParams: { tag_id, chat_id } })
                .then(() => {
                    this.setState({ isRecoverLoading: false, showRecoverConfirm: false, recovered: true });
                })
                .catch((recoverError) => {
                    this.setState({ isRecoverLoading: false, recoverError });
                });
        });
    }

    getEvolutionOptions() {
        const currentTag = this.props.selectedChatItem?.tag_data?.tag;
        const tagEvolutions = this.props.evolutions?.[currentTag] ?? [];

        if (tagEvolutions?.length) {
            let error = null;

            this.request.exec(REQUESTS.TAG_DESCRIPTIONS)
                .then(response => {
                    const migrateOptions = formatMigrateOptions(response, tagEvolutions);
                    this.setState({ migrateError: error, migrateOptions });
                })
                .catch(migrateError => {
                    if (migrateError.message !== 'The user aborted a request.') {
                        error = migrateError;
                    }

                    this.setState({
                        migrateError: error,
                    });
                });
        }
    }

    setJumpModalOpen(isJumpModalOpen) {
        this.setState({ isJumpModalOpen });
    }

    render() {
        const { isChatAdmin, showForceChatClose, ChatJumpToNode, HidePerformChats } = this.props.BlockRules;
        const userId = this.getId(this.props);
        const isChatGo = this.props.selectedChatItem?.tag_data?.external_chat?.external_chat_id;
        const performer_id = this.props.selectedChatItem?.tag_data?.performer;
        const isChatPerformedByUser = this.props.userId === performer_id;
        const isDeferred =
            this.props.selectedChatItem?.container_tag_data ||
            this.props.selectedChatItem?.tag_data?.tag?.includes(DEFERRED_MARK);
        const chat_id = this.props.selectedChatItem?.id;

        return <div className={style.component}>
            {userId &&
            <>
                <ChatInfoCommon userId={userId} selectedChatItem={this.props.selectedChatItem}/>
                <div className={style.content}>

                    <div className={style.controls}>
                        {
                            this.state.errorControl || this.state.migrateError
                                ? <ShortError error={this.state.errorControl || this.state.migrateError}/>
                                : null
                        }

                        {
                            <>
                                {!HidePerformChats && (!isChatPerformedByUser || isChatAdmin)
                                    ? <Button colorType={ButtonTypes.positive}
                                              isLoading={this.state.isWorking}
                                              disabled={!this.getTagId()}
                                              onClick={this.perform.bind(this)}>Забрать чат</Button>
                                    : null
                                }

                                {this.state.migrateOptions?.length
                                    ? <MigrateButton chatItem={this.props.selectedChatItem}
                                                     options={this.state.migrateOptions}
                                                     updateChatItemTag={this.props.updateChatItemTag}/>
                                    : null
                                }

                                {isChatPerformedByUser && isChatGo && showForceChatClose
                                    ? <Button onClick={this.beforeClose
                                        .bind(this, CLOSE_CHAT_ACTIONS.close_ignore_external)}>
                                        Закрыть (без закрытия в такси)
                                    </Button>
                                    : null
                                }

                                {isChatPerformedByUser || isChatAdmin
                                    ? <>
                                        <Button onClick={this.dropPerform.bind(this)}
                                                isLoading={this.state.isWorking}
                                                colorType={ButtonTypes.negative}
                                                basic>Отказаться от чата</Button>
                                        <Button onClick={this.beforeClose.bind(this, CLOSE_CHAT_ACTIONS.close)}
                                                isLoading={this.state.isWorking}
                                                colorType={ButtonTypes.negative}>Закрыть навсегда!!!</Button>
                                        {!isChatGo
                                            ? <Button isLoading={this.state.isWorking}
                                                      onClick={this.beforeClose
                                                          .bind(this, CLOSE_CHAT_ACTIONS.close_light)}>
                                                Скрыть чат из списка до новых сообщений пользователя
                                            </Button>
                                            : null
                                        }
                                        <DeferChatUntil onDeferred={this.onDeferred.bind(this)}
                                                        selectedChatItem={this.props.selectedChatItem}/>
                                    </>
                                    : null
                                }

                                {isDeferred && !this.state.recovered
                                    ? <Button disabled={!this.getTagId() || !this.props.selectedChatItem?.id}
                                              onClick={this.showRecoverConfirm.bind(this, true)}>
                                        Воскресить ✝️
                                    </Button>
                                    : null
                                }

                                <Button isLoading={this.state.isWorking}
                                        onClick={this.showCommentModal.bind(this)}
                                        basic={true}>Комментарий к топику</Button>

                                {ChatJumpToNode
                                    ? <Button onClick={this.setJumpModalOpen.bind(this, true)}
                                              colorType={ButtonTypes.warning}>
                                        Сменить ноду
                                    </Button>
                                    : null
                                }
                            </>
                        }
                    </div>
                </div>
            </>
            }
            {
                this.state.showClassificationDialog
                    ? <ClassificationDialog type={ClassificationType.chat}
                                            onClassified={this.onClassified.bind(this)}
                                            tag_id={this.getTagId()}
                                            user_id={this.props.selectedChatItem?.originator}
                                            chat_id={this.props.selectedChatItem?.id}
                                            onClose={this.showClassificationDialog.bind(this, false)}/>
                    : null
            }
            {
                this.state.showCommentModal
                    ? <AddComment originator={this.props.selectedChatItem?.originator}
                                  chat_id={this.props.selectedChatItem?.id}
                                  onClose={this.showCommentModal.bind(this, false)}/>
                    : null
            }
            {
                this.state.confirmIsOpen
                    ? <Confirm accept={this.state.acceptAction.bind(null)}
                               isWorking={this.state.isWorking}
                               onClose={this.switchConfirm.bind(this, false)}
                               question={this.state.question}
                               error={this.state.errorControl}/>
                    : null
            }
            {
                this.state.showRecoverConfirm
                    ? <Confirm question={'Воскресить чат?'}
                               error={this.state.recoverError}
                               isWorking={this.state.isRecoverLoading}
                               accept={this.onChatRecover.bind(this)}
                               onClose={this.showRecoverConfirm.bind(this, false)}/>
                    : null
            }
            {
                this.state.isJumpModalOpen
                    ? <JumpToNodeModal onClose={this.setJumpModalOpen.bind(this, false)}
                                       chat_id={chat_id}
                                       user_id={userId}/>
                    : null
            }
        </div>;
    }
}
