import React from 'react';

import { Dict } from '../../../../../../types';
import { ONE_SECOND } from '../../../../../constants';
import { getRenderedBBCode } from '../../../../../ui/BBCodeRenderer';
import FormatDate from '../../../../../ui/FormatDate';
import { Link } from '../../../../../ui/Link';
import { Request2 } from '../../../../../utils/request';
import { CommonMessage } from '../../../../ChatsCommon/CommonMessage/component';
import { getImgPrefix } from '../../../../ChatsCommon/TopicView/component';
import { ExpectedActionType, IMessage } from '../../../../ChatsCommon/types';
import { Copy } from '../../../../Copy';
import { FineInfoMessageBlock } from '../../../../Fines/FineMessageType/FineInfoMessage';
import { SimpleError } from '../../../../SimpleError';
import Spin from '../../../../Spin';
import { PDF_ICON, PDF_TYPE, StickerView } from '../../../constants';
import { CHAT_REQUESTS, REQUESTS } from '../../../request';
import ShortSessionInfoBlock from '../../../ShortSessionInfoBlock';
import { MESSAGE_TYPE } from '../../../types';
import * as style from './index.css';

export interface IChatMessageItemProps {
    item: IMessage;
    chat: Dict<any>;
    toBottom: () => {};
    showGallery: (index) => {};
    isSafeMode: boolean;
    isSelected: boolean;
    changeSelectedMessages: (id: number) => void;
    selectedChatItem: any;
}

export const ChatMessageItem = React.memo((props: IChatMessageItemProps) => {
    switch (props.item.type) {
    case MESSAGE_TYPE.PLAINTEXT :
        return <PlainTextMessage {...props}/>;
    case MESSAGE_TYPE.BONUS:
        return <BonusMessage {...props}/>;
    case MESSAGE_TYPE.WELCOME_TEXT:
    case MESSAGE_TYPE.SEPARATOR:
        return <WelcomeText {...props}/>;
    case MESSAGE_TYPE.CREDIT_CARD:
        return <CreditCardText {...props}/>;
    case MESSAGE_TYPE.LOCATION:
        return <LocationMessage {...props}/>;
    case MESSAGE_TYPE.STICKER:
        return <StickerMessage {...props}/>;
    case MESSAGE_TYPE.ORDER:
        return <OrderMessage {...props}/>;
    case MESSAGE_TYPE.MEDIA_RESOURCES:
        return <MediaMessage {...props}/>;
    case MESSAGE_TYPE.USER_DOCUMENTS_PHOTOS:
    case MESSAGE_TYPE.USER_DOCUMENTS_PHOTO:
        return <ImageDocMessage {...props}/>;
    case MESSAGE_TYPE.IMAGE:
        return <ImageMessage {...props}/>;
    case MESSAGE_TYPE.IMAGE_LINK:
        return <ImageLinkMessage {...props}/>;
    case MESSAGE_TYPE.FINE:
        return <FineInfoMessage {...props}/>;
    default:
        return <UnknownMessageType {...props}/>;
    }
});

const WelcomeText = React.memo((props: IChatMessageItemProps) => {
    return <div className={`${style.notes_message}`}>
        <div> {getRenderedBBCode(props.item.text)}</div>
        <div className={style.note_date}>
            <FormatDate value={props.item.timestamp * ONE_SECOND}/>
        </div>
    </div>;
});

export const ExpectedAction = React.memo((props: { expected_action }) => {
    const expectedTypes = [ExpectedActionType.tree, ExpectedActionType.user_message];
    const { expected_action = {} } = props;

    return !expectedTypes.includes(expected_action?.type) && (expected_action?.type || expected_action?.text)
        ? <div className={`${style.expected} ${style.notes_message}`}>
            <strong>{expected_action?.type}</strong>
            {expected_action?.text ? ': ' + expected_action?.text : ''}
        </div>
        : null;
});

const CreditCardText = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        <h4>Банковская карта</h4>
        <div>{props.item.text}</div>
        <Link href={`#/clients/${props.item.author}/billing`} target={'_blank'}>Биллинг</Link>
    </CommonMessage>;
});

const PlainTextMessage = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        {getRenderedBBCode(props.item.text)}
    </CommonMessage>;
});

const BonusMessage = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        <div className={style.bonus}>
            <img src={props.item.bg_link} width={props.item.bg_width} height={props.item.bg_height}/>
            <div className={style.bonus_text}> {getRenderedBBCode(props.item.text)}</div>
        </div>
    </CommonMessage>;
});

const LocationMessage = React.memo((props: IChatMessageItemProps) => {
    const ll = props.item.text.split(',').reverse().join(',');
    const src = `https://static-maps.yandex.ru/1.x/?ll=${ll}&size=350,350&z=16&l=map&pt=${ll},comma`;
    const href = `https://n.maps.yandex.ru/#!/?z=18&ll=${ll}`;
    const onLoad = () => {
        props.toBottom && props.toBottom();
    };

    return <CommonMessage {...props}>
        <Link href={href} target={'_blank'}><img src={src} onLoad={onLoad}/></Link>
    </CommonMessage>;
});

const StickerMessage = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        <Link href={props.item.text} target={'_blank'}>Это стикер</Link>
        <div className={style.sticker_wrapper}>
            <React.Suspense fallback={<Spin/>}>
                <StickerView url={props.item.text}/>
            </React.Suspense>
        </div>
    </CommonMessage>;
});

const ImageMessage = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        {
            props.item.text.split(',').map((item, index) => {
                return <ImageMessageItem key={index}
                                         preview={`${item}`}
                                         image={item}
                                         url
                                         onClick={props.showGallery.bind(null, item)}/>;
            })
        }

    </CommonMessage>;
});

const ImageLinkMessage = React.memo((props: IChatMessageItemProps) => {
    const { link, icon, text } = props.item;
    const pdfProps = { link, icon, text };

    return <CommonMessage {...props}>
        {Render_PDF_View(pdfProps)}
    </CommonMessage>;
});

export const makeSrcForOldDocumentPhoto = (item, id) => {
    const index = item.indexOf('photo_id=');
    let photo_id = '';
    let src = '';
    const photo_id_length = 9;

    if (index > -1) {
        photo_id = item.substr(index + photo_id_length);
    }

    if (photo_id) {
        src = `https://carsharing.yandex-team.ru/py/admin/v1/users/`
            + `${id}/documents/null/photos/`
            + `${photo_id}/`;
    }

    return src;
};

const ImageDocMessage = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        {
            props.item.text.split(',').map((item, index) => {

                const src = makeSrcForOldDocumentPhoto(item, props.item.author);

                return <ImageMessageItem key={index}
                                         preview={`${src}thumbnail/`}
                                         image={`${src}content/`}
                                         url
                                         onClick={props.showGallery.bind(null, src)}/>;
            })
        }

    </CommonMessage>;
});

const MediaMessage = React.memo((props: IChatMessageItemProps) => {
    const ct = props.item.content_types.filter(el => el);
    const isPDF = props.item.content_types?.includes(PDF_TYPE);

    const pdfProps = {
        link: `${getImgPrefix()}/${props.item.cached_images[0]}`,
        icon: PDF_ICON,
        text: props.item.text,
    };

    return isPDF
        ? <CommonMessage {...props}>{Render_PDF_View(pdfProps)}</CommonMessage>
        : <CommonMessage {...props}>
            <i>Загружено файлов: {ct.length} / {props.item.content_types.length}</i>
            <div className={style.media_container}>
                {
                    props.item.content_types.map((item, index) => {
                        if (item) {
                            if (item.includes('image')) {
                                return <ImageMessageItem key={props.item.cached_images[index]}
                                                         preview={props.item.cached_previews[index]}
                                                         image={props.item.cached_images[index]}
                                                         content_type={item}
                                                         onClick={props.showGallery
                                                             .bind(null, props.item.cached_images[index])}/>;
                            }

                            if (item.includes('video')) {
                                return <VideoMessageItem key={props.item.cached_images[index]}
                                                         content_type={item}
                                                         preview={props.item.cached_previews[index]}
                                                         image={props.item.cached_images[index]}/>;
                            }

                            return <div key={index}>{item}</div>;
                        }

                        return <EmptyMedia key={index}/>;

                    })
                }
            </div>
        </CommonMessage>;
});

const VideoMessageItem = React.memo((props: { preview: string; image: string; content_type?: string }) => {
    const url = `${getImgPrefix()}/${props.image}`;

    return <div className={`${style.video_item} ${style.media_video}`}>
        <video controls loop muted src={url}/>
    </div>;
});

const ImageMessageItem = React.memo((props: {
    preview: string;
    image: string;
    onClick: () => void;
    url?: boolean;
    content_type?: string;
}) => {
    const prefix = getImgPrefix();

    const preview = props.preview || props.image;
    const previewUrl = !props.url ? `${prefix}/${preview}` : preview;

    const image = props.image || props.preview;
    const imageUrl = !props.url ? `${prefix}/${image}` : image;

    return <div className={style.imageBlock}>
        <div className={`${style.media_item} ${style.media_img}`}
             onClick={props.onClick}
             style={{ backgroundImage: `url(${previewUrl})` }}/>
        <div><i>content_type: <strong>{props.content_type}</strong></i></div>
        <Copy text={imageUrl} internalTooltipId={'PhotoLink'}>{'Копировать ссылку'}</Copy>
    </div>;
});

const EmptyMedia = React.memo(() => {
    return <div className={`${style.media_item} ${style.empty_media}`}/>;
});

class OrderMessage extends React.Component<IChatMessageItemProps> {
    request = new Request2({ requestConfigs: CHAT_REQUESTS });
    state: any = {
        isLoading: false,
        data: null,
        error: null,
    };

    componentDidMount(): void {
        this.getSession();
    }

    getSession() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request
                .exec(REQUESTS.SESSION_HISTORY, { queryParams: { session_id: this.props.item.text } })
                .then(response => {
                    this.setState({
                        isLoading: false,
                        error: null,
                        data: response,
                    }, () => {
                        this.props.toBottom();
                    });
                })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    });
                });
        });
    }

    render() {
        return <CommonMessage {...this.props}>
            <div className={style.order_message_title}>
                <strong>Поездка:</strong> <Copy>{this.props.item.text}</Copy></div>
            {
                this.state.isLoading ? <Spin/>
                    : this.state.error
                        ? <SimpleError error={this.state.error}/>
                        : this.state.data &&
                    <ShortSessionInfoBlock sessionInfo={this.state.data}/>
            }
        </CommonMessage>;
    }
}

const UnknownMessageType = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        UNREC: {props.item.type}: {props.item.text}
    </CommonMessage>;
});

const FineInfoMessage = React.memo((props: IChatMessageItemProps) => {
    return <CommonMessage {...props}>
        <FineInfoMessageBlock {...props} />
    </CommonMessage>;
});

interface IClassificationMessageProps {
    classification: string;
    comment: string;
    timestamp: number;
    author?: string;
}

export const ClassificationMessage = React.memo((props: IClassificationMessageProps) => {
    return <div className={`${style.classification_message} ${style.notes_message}`}>
        <div>Комментарий к классификации
            <span className={style.classification}>{` ` + props.classification}:</span>
        </div>
        <div><b>{props.comment}</b></div>
        <div className={style.author}>{props.author} {<FormatDate value={props.timestamp * ONE_SECOND}/>}</div>
    </div>;
});

const Render_PDF_View = (props: { link: string; icon: string | undefined; text: string }) => {
    const { link, icon, text } = props;

    return <Link href={link} target={'_blank'} className={style.img_link_wrapper}>
        {icon ? <img src={icon} alt={'pdf'}/> : <div/>}

        <div>
            {getRenderedBBCode(text)}
        </div>
    </Link>;
};
