import * as React from 'react';

import { EMPTY_DATA } from '../../../../constants';
import { UserInfoHandler } from '../../../../models/user';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { getImgPrefix } from '../../../ChatsCommon/TopicView/component';
import { NChats } from '../../../ChatsCommon/types';
import { ErrorSource, logError } from '../../../Content/initErrorCounter';
import { PictureGallery } from '../../../PictureGallery';
import { SimpleError } from '../../../SimpleError';
import { CHAT_REQUESTS } from '../../request';
import { MESSAGE_TYPE } from '../../types';
import ChatMessageItem from './ChatMessageItem';
import { ClassificationMessage, ExpectedAction, makeSrcForOldDocumentPhoto } from './ChatMessageItem/component';
import { IChatWindowContentMapStateToProps } from './index';
import * as style from './index.css';
import ISelectedChatData = NChats.ISelectedChatData;

interface IChatWindowContentProps extends IChatWindowContentMapStateToProps {
    selectedChatData: ISelectedChatData;
    isSafeMode: boolean;
    categorizations: any[];
    selectedMessages: number[];
    changeSelectedMessages: (id: number) => void;
    tagData: any;
}

export class ChatWindowContent extends React.Component<IChatWindowContentProps> {
    request = new Request2({ requestConfigs: CHAT_REQUESTS });
    chatWindow: React.RefObject<any>;
    state = {
        error: null,
        isWorking: false,
        showImgIndex: undefined,
        pictures: [],
        extraError: null,
        checkScroll: true,
    };

    showGallery(id) {
        let showImgIndex = id;
        if (id !== undefined) {
            for (let i = 0; i < this.state.pictures?.length; i++) {
                const el: any = this.state.pictures[i];
                if (el && el.link && el.link.includes(id)) {
                    showImgIndex = i;
                    break;
                }
            }
        }

        this.setState({
            showImgIndex,
        });
    }

    constructor(props) {
        super(props);
        this.chatWindow = React.createRef();
    }

    componentDidMount(): void {
        this.prepareImgForGallery();
        this.toBottom();
    }

    componentDidUpdate(prevProps: Readonly<IChatWindowContentProps>, prevState: Readonly<{}>, snapshot?: any): void {
        const scrollElement = this.chatWindow.current;
        const needScroll = scrollElement?.scrollHeight > scrollElement?.offsetHeight;
        const messages = this.props.selectedChatData?.messages ?? [];
        const prevMessages = prevProps.selectedChatData?.messages ?? [];

        if (this.state.checkScroll && needScroll && !scrollElement.scrollTop) {
            this.toBottom();
        }

        if (!isObjectEqual(messages, prevMessages)) {
            this.prepareImgForGallery();
            this.toBottom();
        }
    }

    prepareImgForGallery() {
        const prefix = getImgPrefix();

        this.props.selectedChatData && this.setState({
            pictures: this.props.selectedChatData?.messages
                && Array.isArray(this.props.selectedChatData.messages)
                && this.props.selectedChatData.messages.reduce((_p, _c) => {
                    if (_c.content_types && _c.content_types.some(i => i && i.includes('image') || false)
                        && _c.cached_images && Array.isArray(_c.cached_images)) {
                        return [..._p, ..._c.cached_images.map(img => ({ link: `${prefix}/${img}` }))];
                    }

                    if (
                        [MESSAGE_TYPE.USER_DOCUMENTS_PHOTO, MESSAGE_TYPE.USER_DOCUMENTS_PHOTOS]
                            .includes(_c.type)) {
                        return [..._p, ..._c.text.split(',')
                            .map(link => (
                                {
                                    link: `${makeSrcForOldDocumentPhoto(link, _c.author)}content/`,
                                }
                            ))];
                    }

                    if ([MESSAGE_TYPE.IMAGE].includes(_c.type)) {
                        return [..._p, ..._c.text.split(',')
                            .map(link => ({ link }))];
                    }

                    return _p;
                }, []),
        });
    }

    toBottom() {
        const scrollElement = this.chatWindow.current;

        scrollElement
            ? scrollElement.scrollTop = scrollElement.scrollHeight - scrollElement.offsetHeight
            : null;

        if (scrollElement.scrollTop) {
            this.setState({ checkScroll: false });
        }
    }

    shouldComponentUpdate(
        nextProps: Readonly<IChatWindowContentProps>,
        nextState: Readonly<{}>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);
        this.setState({
            extraError: error,
        });
    }

    addClassificationMessages() {
        const messages: any = [];

        this.props.categorizations?.forEach(el => {
            const author = this.props.selectedChatData?.users?.[el?.operator_id] || {};
            messages.push({
                classification: el?.category?.meta?.label ?? EMPTY_DATA,
                comment: el?.comment,
                timestamp: el?.timestamp,
                author: UserInfoHandler.getPrintName.call(author),
            });
        });

        return messages;
    }

    render() {
        const classificationMessages = this.addClassificationMessages();

        return <div className={`${style.component}`} ref={this.chatWindow}>
            {
                this.state.showImgIndex !== undefined &&
                <PictureGallery initialIndex={this.state.showImgIndex || 0}
                                pictures={this.state.pictures}
                                onClose={this.showGallery.bind(this, undefined)}/>
            }
            {this.state.extraError
                ? <SimpleError error={this.state.extraError}/>
                : <>
                    {this.props.selectedChatData?.messages?.map((item) => {
                        return <ChatMessageItem key={item.id}
                                                item={item}
                                                isSelected={this.props.selectedMessages?.includes(item.id)}
                                                selectedChatItem={this.props.selectedChatItem || {}}
                                                changeSelectedMessages={this.props.changeSelectedMessages.bind(this)}
                                                toBottom={this.toBottom.bind(this)}
                                                showGallery={this.showGallery.bind(this)}
                                                chat={this.props.selectedChatData}
                                                isSafeMode={this.props.isSafeMode}/>;
                    })}

                    {classificationMessages.length
                        ? classificationMessages.map((el, ind) => {
                            return <ClassificationMessage {...el} key={ind}/>;
                        })
                        : null
                    }
                </>
            }
            <ExpectedAction expected_action={this.props.selectedChatData?.expected_action}/>
        </div>;
    }
}
