import * as React from 'react';

import ImportantIcon from '../../../../../svg-components/chat-important.component.svg';
import CopyIcon from '../../../../../svg-components/copy.component.svg';
import HideIcon from '../../../../../svg-components/hide.component.svg';
import HistoryIcon from '../../../../../svg-components/history.component.svg';
import LinkIcon from '../../../../../svg-components/link.component.svg';
import MutedIcon from '../../../../../svg-components/muted.component.svg';
import { EMPTY_DATA } from '../../../../constants';
import { UserInfoHandler } from '../../../../models/user';
import { Confirm, Window } from '../../../../ui/FullModal';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { deepCopy } from '../../../../utils/utils';
import { ChatHeaderClassification } from '../../../ChatsCommon/ChatHeaderClassification';
import { chatLink } from '../../../ChatsCommon/helpers';
import { NChats } from '../../../ChatsCommon/types';
import { Copy } from '../../../Copy';
import { ChatHistoryView } from '../../ChatHistoryView/component';
import { CHAT_REQUESTS, REQUESTS } from '../../request';
import * as chatStyle from '../index.css';
import { IChatWindowHeaderMapDispatchToProps, IChatWindowHeaderMapStateToProps } from './index';
import * as style from './index.css';
import ICategory = NChats.ICategory;

interface IChatWindowHeaderProps extends IChatWindowHeaderMapStateToProps, IChatWindowHeaderMapDispatchToProps {
    getImgLinks: string;
    categorizations: ICategory[];
    updateCategorizations: () => void;
    changeMode: (isSafeMode: boolean) => void;
}

export class ChatWindowHeader extends React.Component<IChatWindowHeaderProps> {
    request = new Request2({ requestConfigs: CHAT_REQUESTS });
    state = {
        showConfirm: false,
        error: null,
        isWorking: false,
        showChatHistory: false,
        isSafeMode: false,

        openImportantWarningWindow: false,
        showChatImportantConfirm: false,
        chatImportantSpin: false,
        chatImportantError: null,
    };

    shouldComponentUpdate(nextProps: Readonly<IChatWindowHeaderProps>, nextState, nextContext: any): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    componentDidUpdate(prevProps: Readonly<IChatWindowHeaderProps>, prevState: Readonly<{}>, snapshot?: any): void {
        if (!isObjectEqual(this.props, prevProps)) {
            this.setNewMode(false);
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getRequestIds() {
        return {
            user_id: this.props.selectedChatItem?.originator,
            chat_id: this.props.selectedChatItem?.id,
        };
    }

    toggleMute(showConfirm) {
        this.setState({
            showConfirm,
        });
    }

    accept(isMuted, selectedChat) {
        const tag_id = this.getTagId();
        const muted = !selectedChat?.tag_data?.muted;
        const tag = Object.assign({}, selectedChat?.tag_data, { tag_id, muted });
        const chat = Object.assign({}, selectedChat, { tag_data: tag });

        this.setState({
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.UPDATE_USER_TAG, {
                queryParams: { tag_id },
                body: tag,
            })
                .then(() => {
                    this.setState({
                        showConfirm: false,
                        isWorking: false,
                        error: null,
                    });
                    this.props.updateChatItemTag?.(chat, tag);
                })
                .catch(error => {
                    this.setState({
                        isWorking: false,
                        error,
                    });
                });
        });
    }

    showChatHistory(showChatHistory) {
        this.setState({ showChatHistory });
    }

    getTagId() {
        return this.props.selectedChatItem?.tag_data?.tag_id || this.props.selectedChatItem?.tag_id;
    }

    changeMode() {
        const newMode = !this.state.isSafeMode;
        this.setNewMode(newMode);
    }

    setNewMode(isSafeMode: boolean) {
        this.setState({ isSafeMode });
        this.props.changeMode(isSafeMode);
    }

    chatImportantToggle(showChatImportantConfirm: boolean) {
        this.setState({ showChatImportantConfirm, chatImportantError: null });
    }

    beforeImportantRequest() {
        const tag_id = this.getTagId();
        if (tag_id) {
            const chat_data = deepCopy(this.props.selectedChatItem);
            const tag_data = deepCopy(chat_data.tag_data) ?? {};
            let new_chat_meta = deepCopy(tag_data?.chat_meta);
            tag_data.tag_id = tag_id;

            if (typeof new_chat_meta === 'string') {
                new_chat_meta = {
                    important: true,
                };
            } else {
                new_chat_meta.important = !new_chat_meta?.important;
            }

            const tag = Object.assign({}, tag_data, { chat_meta: new_chat_meta });
            const chat = Object.assign({}, chat_data, { tag_data: tag });

            return { tag_id, tag, chat };
        }

        return null;

    }

    acceptChatImportant() {
        const data = this.beforeImportantRequest();
        const needQuery = data && data.tag && data.tag_id;

        needQuery && this.setState({
            chatImportantSpin: true,
        }, () => {
            this.request.exec(REQUESTS.UPDATE_USER_TAG, {
                queryParams: { tag_id: data?.tag_id },
                body: data?.tag,
            })
                .then(() => {
                    this.setState({
                        showChatImportantConfirm: false,
                        chatImportantSpin: false,
                    });
                    this.props.updateChatItemTag?.(data?.chat, data?.tag);
                })
                .catch(chatImportantError => {
                    this.setState({
                        chatImportantSpin: false,
                        chatImportantError,
                    });
                });
        });

        if (!needQuery) {
            this.chatImportantWarn(true);
            this.chatImportantToggle(false);
        }
    }

    chatImportantWarn(openImportantWarningWindow) {
        this.setState({ openImportantWarningWindow });
    }

    render() {
        const {
            originatorInfo,
            selectedChatItem,
            getImgLinks,
            selectedChatDisplayName,
            updateCategorizations,
        } = this.props;
        const originatorName = UserInfoHandler.getPrintName.call(originatorInfo);
        const isMuted = selectedChatItem?.tag_data?.muted;

        const classificationData: ICategory[] = this.props.categorizations || [];
        const classificationParams = {
            tag_id: this.getTagId(),
            chat_id: selectedChatItem?.id,
            user_id: selectedChatItem?.originator,
        };

        const imgLinks = getImgLinks;
        const textForST = `https://carsharing.yandex-team.ru/beta/#/chats`
            + `?user_id=${selectedChatItem?.originator}`
            + `&chat_id=${selectedChatItem?.id}\n`
            + `<{Фотографии из чата с клиентом\n`
            + `${imgLinks}`
            + `}>`;

        const isImportant = selectedChatItem?.tag_data?.chat_meta?.important;
        const displayName = selectedChatDisplayName || selectedChatItem?.tag_data?.tag;

        return <div className={`${style.component}`}>
            {this.state.showConfirm
                ? <Confirm accept={this.accept.bind(this, isMuted, selectedChatItem)}
                           error={this.state.error}
                           isWorking={this.state.isWorking}
                           onClose={this.toggleMute.bind(this, false)}
                           question={isMuted ? 'Включить ОРАЛО! (мониторинг)' : 'Выключить ОРАЛО! (мониторинг)'}/>
                : null
            }

            {this.state.showChatImportantConfirm
                ? <Confirm accept={this.acceptChatImportant.bind(this)}
                           error={this.state.chatImportantError}
                           isWorking={this.state.chatImportantSpin}
                           onClose={this.chatImportantToggle.bind(this, false)}
                           question={isImportant ? 'Удалить чат из важных?' : 'Пометить чат как важный?'}/>
                : null
            }

            <div className={style.originator}>
                <MutedIcon className={`${style.muted_icon} ${isMuted ? style.muted : ''}`}
                           title={`is muted: ${isMuted}`}
                           onClick={this.toggleMute.bind(this, true)}/>
                <HistoryIcon className={`${style.history_icon} `}
                             title={`История действий с чатом`}
                             onClick={this.showChatHistory.bind(this, true)}/>
                <ImportantIcon className={`${style.important_chat} ${isImportant ? style.active_mode : style.passive_mode}`}
                               title={'Пометить как важный'}
                               onClick={this.chatImportantToggle.bind(this, true)}/>
                <HideIcon className={`${style.mode} ${this.state.isSafeMode ? style.active_mode : style.passive_mode}`}
                          title={'Безопасный режим'}
                          onClick={this.changeMode.bind(this)}/>
                <span className={`${style.short_name} ${this.state.isSafeMode ? chatStyle.safe_mode : null}`}>
                    <Copy text={originatorName}>{originatorName}</Copy>
                </span>
            </div>

            <ChatHeaderClassification classificationData={classificationData}
                                      classificationParams={classificationParams}
                                      getClassification={updateCategorizations.bind(this)}/>

            <div>
                <i>
                    {displayName ? `${displayName}, ` : ''}
                    chat_id: {selectedChatItem?.id ?? EMPTY_DATA}
                </i>
            </div>

            {
                this.state.showChatHistory
                && <ChatHistoryView user_id={selectedChatItem?.originator}
                                    chat_id={selectedChatItem?.id}
                                    tag_id={this.getTagId()}
                                    onClose={this.showChatHistory.bind(this, false)}/>
            }
            {this.state.openImportantWarningWindow
                ? <Window title={'Упс'} onClose={this.chatImportantWarn.bind(this, false)}>
                    Такой чат нельзя пометить важным:(
                </Window>
                : null
            }

            <div className={`${style.share}`}>
                <Copy text={chatLink(selectedChatItem, false, this.getTagId())}>
                    <LinkIcon className={style.share_chat_icon}/>
                </Copy>
            </div>

            {imgLinks.length ? <div className={`${style.share} ${style.share_img}`}>
                <Copy text={textForST}>
                    <CopyIcon className={style.copy_chat_icon}/>
                </Copy>
            </div>
                : null
            }

        </div>;
    }
}
