import React, { ReactElement } from 'react';

import { GetClassification, IGetClassificationExport } from '../../../../decorators/GetClassification';
import DollarIcon from '../../../../svg-components/dollar.component.svg';
import FileIcon from '../../../../svg-components/file.component.svg';
import FineIcon from '../../../../svg-components/fine.component.svg';
import PdfIcon from '../../../../svg-components/pdf.component.svg';
import SessionIcon from '../../../../svg-components/session.component.svg';
import StickerIcon from '../../../../svg-components/sticker-btn.component.svg';
import { ClassificationType } from '../../../../types';
import { CLIENT_ERRORS_CODE_START, TIMERS } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import { Confirm } from '../../../ui/FullModal';
import { IRichEditorControlItem, RichEditor } from '../../../ui/RichEditor';
import { CSSettingsItems, CustomSettings } from '../../../utils/customSettings';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { MessagesByTopic } from '../../../utils/localStorage/localStorage';
import { AbortError, Request2 } from '../../../utils/request';
import { BonusControlDialog } from '../../BonusControlDialog';
import { getTagId } from '../../ChatsCommon/chatUtils';
import { chatLink } from '../../ChatsCommon/helpers';
import { getImgPrefix, SessionDialog } from '../../ChatsCommon/TopicView/component';
import { ExpectedActionType, NChats } from '../../ChatsCommon/types';
import { ErrorSource, logError } from '../../Content/initErrorCounter';
import FileChooserDialog from '../../FileChooserDialog';
import { FineDialogModal } from '../../Fines/FineMessageType/FineDialogModal';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import ClassificationDialog from '../ClassificationDialog';
import { PDF_ICON, VALIDATION_CONFIRM_QUESTION } from '../constants';
import * as chatStyle from '../index.css';
import { CHAT_REQUESTS, REQUESTS } from '../request';
import StickerDialog from '../Sticker/StickerDialog';
import { MESSAGE_TYPE } from '../types';
import { getChatMetaInfo, matchWelcomeTime, stringHasNumber } from '../utils';
import ChatWindowContent from './ChatWindowContent';
import ChatWindowHeader from './ChatWindowHeader';
import ForwardWindow from './ForwardWindow/component';
import { ChatWindowMapDispatchToProps, ChatWindowMapStateToProps } from './index';
import * as style from './index.css';
import IChatListItem = NChats.IChatListItem;
import ISelectedChatData = NChats.ISelectedChatData;

interface IChatWindowProps extends ChatWindowMapStateToProps, ChatWindowMapDispatchToProps, IGetClassificationExport {
}

interface IChatWindowState {
    isEmpty: boolean;
    selectedChatData: ISelectedChatData;
    isLoading: boolean;
    error: Error | null;
    filesError: Error | null;
    showBonusDialog: boolean;
    showSessionDialog: boolean;
    showStickerDialog: boolean;
    showFineDialog: boolean;
    showClassificationModal: boolean;
    isUploading: boolean;
    isSafeMode: boolean;
    validationConfirmOpen: boolean;
    confirmValidation: boolean;
    showForwardWindow: boolean;
    selectedMessages: number[];
    initText: string;
    fileWindowState: FileWindowStateType;
}

type FileWindowStateType = 'media' | 'pdf' | null;

const FILE_COUNT_LIMIT = 5;

@GetClassification()
export class ChatWindow extends React.Component<IChatWindowProps, IChatWindowState> {
    richEditor: React.RefObject<RichEditor>;
    state: IChatWindowState = {
        isEmpty: true,
        selectedChatData: {} as ISelectedChatData,
        isLoading: false,
        error: null,
        filesError: null,
        showBonusDialog: false,
        showSessionDialog: false,
        showStickerDialog: false,
        showFineDialog: false,
        showClassificationModal: false,
        isUploading: false,
        isSafeMode: false,
        validationConfirmOpen: false,
        confirmValidation: false,
        fileWindowState: null,

        selectedMessages: [],
        showForwardWindow: false,
        initText: '',
    };
    timer: any;
    request = new Request2({ requestConfigs: CHAT_REQUESTS });
    requestMessage = new Request2({ requestConfigs: CHAT_REQUESTS });
    requestClassification = new Request2({ requestConfigs: CHAT_REQUESTS });
    mbt = new MessagesByTopic();
    message: string;
    autoWelcomes: boolean;

    constructor(props) {
        super(props);
        this.richEditor = React.createRef();
        const cs = new CustomSettings()?.getSettings() || {};
        this.autoWelcomes = cs[CSSettingsItems.showWelcomes];
    }

    componentDidMount(): void {
        this.setTextFromStorage();
        this.getData();
    }

    shouldComponentUpdate(
        nextProps: Readonly<IChatWindowProps>,
        nextState: Readonly<IChatWindowState>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    componentDidUpdate(prevProps: Readonly<IChatWindowProps>, prevState: Readonly<{}>, snapshot?: any): void {
        const tag_id = this.getTagId();
        const originator = this.props.selectedChatItem?.originator;
        const prev_tag_id = getTagId(prevProps.selectedChatItem);
        const prev_originator = prevProps.selectedChatItem?.originator;
        const chat_id = this.props.selectedChatItem?.id;
        const prev_chat_id = prevProps.selectedChatItem?.id;

        const isTagChanged = tag_id && tag_id !== prev_tag_id;
        const isOriginatorChanged = originator && originator !== prev_originator;
        const isChatIdChanged = chat_id !== prev_chat_id;

        if (isTagChanged || isOriginatorChanged || isChatIdChanged) {
            this.requestClassification.abort();
            this.setState({
                selectedChatData: {} as ISelectedChatData,
                selectedMessages: [],
                initText: '',
                error: null,
            }, () => {
                this.setTextFromStorage();
                this.getData();
                this.props.getClassification?.(tag_id);
            });
        }
    }

    componentWillUnmount(): void {
        clearInterval(this.timer);
        this.request.abort();
        this.requestMessage.abort();
        this.requestClassification.abort();
        this.removeMessageFromStorage();
    }

    componentDidCatch(error: Error, errorInfo: React.ErrorInfo): void {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);
        this.setState({ error });
    }

    getTagId() {
        return this.props.selectedChatItem?.tag_data?.tag_id || this.props.selectedChatItem?.tag_id;
    }

    setTextFromStorage() {
        return this.props.selectedChatItem?.id
            && this.mbt.getText(this.props.selectedChatItem.id) || '';
    }

    onRichEditorChange(text) {
        const id = this.props.selectedChatItem && this.props.selectedChatItem.id;
        id && this.mbt.setText({ id, text });
    }

    getChatData(chat: IChatListItem) {
        this.setState({
            isLoading: !Object.keys(this.state.selectedChatData).length,
        }, () => {
            this.request.abort();
            this.request.exec(REQUESTS.HISTORY, {
                queryParams: {
                    chat_id: chat.id,
                    user_id: chat.originator,
                },
            })
                .then(response => {
                    this.setState({
                        error: null,
                        isLoading: false,
                        selectedChatData: {
                            expected_action: response.expected_action,
                            messages: response.messages,
                            user_last_viewed: response.user_last_viewed,
                            users: response.users && Array.isArray(response.users)
                                && response.users.reduce((_p, _c) => {
                                    if (!_p.hasOwnProperty(_c.id)) {
                                        _p[_c.id] = _c;
                                    }

                                    return _p;
                                }, {}),
                        },
                    }, () => {
                        this.update();
                        if (this.props.selectedChatItem?.id
                            && !this.mbt.getText(this.props.selectedChatItem.id)
                            && !this.state.initText
                            && this.autoWelcomes) {
                            this.setInitText();
                        }
                    });
                })
                .catch(error => {
                    const status = error.response?.status;
                    const isClientError = (status)?.toString()?.[0] == CLIENT_ERRORS_CODE_START;

                    if (error.name !== AbortError) {
                        this.setState((prev: IChatWindowState) => {
                            const newState: any = {
                                isLoading: false,
                            };

                            const isNewError = prev.error?.message !== error?.message;

                            if (isNewError) {
                                newState.error = error;
                            }

                            return newState;
                        }, () => {
                            !isClientError && this.update();
                        });
                    } else {
                        this.setState({ isLoading: false });
                    }
                });
        });
    }

    setInitText() {
        const { welcomes, waitingTime, apologizes } = this.props.settings;
        const originator = this.props.selectedChatItem?.originator;
        const messages = this.state.selectedChatData?.messages;
        const { isLongWaiting, isAnswered } = getChatMetaInfo(originator, messages, waitingTime);

        let welcomeText;
        let name;
        if (!isAnswered) {
            const variantsCount = 2;
            const welcomeInd = Math.floor(Math.random() * variantsCount);
            welcomeText = welcomeInd > 0
                ? matchWelcomeTime(welcomes)
                : welcomes?.[welcomes?.length - 1]?.welcome;
            name = this.props.users?.find(el => el.id === originator)?.first_name;
        }

        let apologize;
        if (isLongWaiting) {
            const apologizeInd = Math.floor(Math.random() * apologizes?.length);
            apologize = `${apologizes[apologizeInd]}.\n`;
        }

        const userName = name ? `, ${name}.` : '.';
        const initWelcome = `${welcomeText ? `${welcomeText}${userName} ` : ''}`;
        this.setState({
            initText: `${initWelcome}${apologize ?? ''}`,
        });
    }

    update() {
        clearTimeout(this.timer);
        this.timer = setTimeout(() => {
            this.getData();
        }, TIMERS.UPDATE_SELECTED);
    }

    getData() {
        if (this.props.selectedChatItem && this.props.selectedChatItem.id) {
            this.setState({
                isEmpty: false,
            }, () => {
                this.props.selectedChatItem && this.getChatData(this.props.selectedChatItem);
            });
        } else {
            this.setState({
                isEmpty: true,
            });
        }
    }

    removeMessageFromStorage() {
        const id = this.props.selectedChatItem && this.props.selectedChatItem.id;
        this.mbt.remove(id);
    }

    disableRichEditor(state: boolean) {
        this.richEditor && this.richEditor.current && this.richEditor.current.disable(state);
    }

    setStatusText(statusText: ReactElement | string, options?: any) {
        this.richEditor && this.richEditor.current && this.richEditor.current.setStatusText(statusText, options);
    }

    setReachEditText(text: string) {
        this.richEditor && this.richEditor.current && this.richEditor.current.setText(text);
    }

    onSendText(options, message) {
        const {
            message_type = MESSAGE_TYPE.PLAINTEXT,
            statusText = 'Отправка...',
            customHandler = false,
        } = options || {};

        let isValidMessage = true;
        if (message_type === MESSAGE_TYPE.PLAINTEXT) {
            isValidMessage = !stringHasNumber(message);
        }

        if (isValidMessage || this.state.confirmValidation) {
            this.setStatusText(statusText);
            this.disableRichEditor(true);

            this.requestMessage.exec(REQUESTS.SEND_MESSAGE, {
                queryParams: {
                    user_id: this.props.selectedChatItem && this.props.selectedChatItem.originator,
                    chat_id: this.props.selectedChatItem && this.props.selectedChatItem.id,
                },
                body: {
                    message,
                    message_type,
                },
            })
                .then(() => {
                    this.removeMessageFromStorage();
                    this.setStatusText('');
                    !customHandler && this.setReachEditText('');
                    this.disableRichEditor(false);
                    this.resetValidation();
                })
                .catch(error => {
                    this.disableRichEditor(false);
                    this.setStatusText('', { error: true, errorData: error });
                    this.resetValidation();
                });
        } else {
            this.openValidationConfirm(true);
        }
    }

    openValidationConfirm(validationConfirmOpen) {
        this.setState({ validationConfirmOpen });
    }

    acceptValidationConfirm() {
        this.setState({
            validationConfirmOpen: false,
            confirmValidation: true,
        }, () => {
            const text = this.richEditor?.current?.state?.text || '';
            this.onSendText({}, text);
        });
    }

    resetValidation() {
        this.setState({ confirmValidation: false });
    }

    showBonusDialog(state) {
        this.setState({
            showBonusDialog: state,
        });
    }

    showSessionDialog(state) {
        this.setState({
            showSessionDialog: state,
        });
    }

    showStickerDialog(state) {
        this.setState({
            showStickerDialog: state,
        });
    }

    showChoseFileDialog(state: FileWindowStateType) {
        this.setState({
            fileWindowState: state,
            filesError: null,
        });
    }

    showFineDialog(state) {
        this.setState({
            showFineDialog: state,
        });
    }

    uploadFiles(files) {
        this.setState({
            isUploading: true,
            filesError: null,
        }, () => {
            this.requestMessage.exec(REQUESTS.REGISTER_RESOURCE, {
                queryParams: {
                    user_id: this.props.selectedChatItem && this.props.selectedChatItem.originator,
                    chat_id: this.props.selectedChatItem && this.props.selectedChatItem.id,
                },
                body: {
                    message: '',
                    attachments: files.map(f => f.name),
                },
            })
                .then(() => {
                    const all = files.map((f: File) => {
                        const type = f.type;
                        const reader = new FileReader();
                        reader.readAsArrayBuffer(f);
                        reader.onload = () => {
                            this.requestMessage.exec(REQUESTS.UPLOAD_RESOURCE, {
                                queryParams: {
                                    user_id: this.props.selectedChatItem && this.props.selectedChatItem.originator,
                                    chat_id: this.props.selectedChatItem && this.props.selectedChatItem.id,
                                    resource_id: f.name,
                                },
                                file: reader.result,
                                headers: {
                                    'Content-Type': type,
                                },
                            });
                        };
                    });

                    Promise.all(all)
                        .then(() => {
                            this.setState({
                                isUploading: false,
                                fileWindowState: null,
                            });
                        })
                        .catch(error => {
                            this.setState({
                                filesError: error,
                                isUploading: false,
                            });
                        });

                })
                .catch((error) => {
                    this.setState({
                        filesError: error,
                        isUploading: false,
                    });
                });
        });
    }

    uploadPdf(pdfData) {
        this.setState({
            isUploading: true,
        }, () => {
            this.request.exec(REQUESTS.SEND_MESSAGE, {
                queryParams: {
                    user_id: this.props.selectedChatItem && this.props.selectedChatItem.originator,
                    chat_id: this.props.selectedChatItem && this.props.selectedChatItem.id,
                },
                body: {
                    link: pdfData.link,
                    message_type: MESSAGE_TYPE.IMAGE_LINK,
                    icon: PDF_ICON,
                    message: pdfData.name,
                },
            })
                .then(() => {
                    this.setState({
                        isUploading: false,
                        fileWindowState: null,
                    });
                })
                .catch(error => {
                    this.setState({
                        isUploading: false,
                        filesError: error,
                    });
                });
        });
    }

    getImgLinks() {
        const mediaResources = this.state.selectedChatData
            ?.messages?.filter?.(item => item.type === MESSAGE_TYPE.MEDIA_RESOURCES
                && item.author === this.props.selectedChatItem?.originator,
            ) || [];
        const img: string[] = [];

        mediaResources.forEach(item => {
            const text = item?.text?.split(',');
            item?.content_types?.forEach(((type, index) => {
                if (type?.includes('image')) {
                    img.push(`0x0:${getImgPrefix()}/${item.author}/${text[index]}`);
                }
            }));
        });

        return img.join('\n');
    }

    changeMode(isSafeMode: boolean) {
        this.setState({ isSafeMode });
    }

    changeSelectedMessages(id: number) {
        const messages = new Set(this.state.selectedMessages);

        messages.has(id) ? messages.delete(id) : messages.add(id);

        this.setState({ selectedMessages: Array.from(messages) });
    }

    showForwardWindow(showForwardWindow: boolean) {
        this.setState({ showForwardWindow });
    }

    cancelSelect() {
        this.setState({ selectedMessages: [] });
    }

    closeClassificationModal() {
        this.setState({ showClassificationModal: false });
    }

    onClassified() {
        this.closeClassificationModal();
        this.sendAndHide();
    }

    onSendAndHide(message) {
        this.message = message;

        if (!this.props.classificationData?.length) {
            this.setState({ showClassificationModal: true });
        } else {
            this.sendAndHide();
        }
    }

    sendAndHide() {
        const statusText = 'Отправка...';
        this.setStatusText(statusText);
        this.disableRichEditor(true);

        this.requestMessage.exec(REQUESTS.DROP_PERFORM_NEW, {
            queryParams: {
                tag_id: this.getTagId(),
                is_finishing: false,
            },
            body: { message: this.message },
        })
            .then(() => {
                this.removeMessageFromStorage();
                this.setStatusText('');
                this.setReachEditText('');
                this.disableRichEditor(false);
                this.props.cancelChatCallBack?.();
            })
            .catch(error => {
                this.disableRichEditor(false);
                this.setStatusText('', { error: true, errorData: error });
            });
    }

    getAdditionalControls() {
        return [
            {
                title: <DollarIcon/>,
                tooltip: 'Бонусы',
                handle: this.showBonusDialog.bind(this, true), isActive: true,
            },
            {
                title: <SessionIcon/>,
                tooltip: 'Поездка',
                handle: this.showSessionDialog.bind(this, true), isActive: true,
            },
            {
                title: <StickerIcon/>,
                tooltip: 'Стикеры',
                handle: this.showStickerDialog.bind(this, true), isActive: true,
            },
            {
                title: <FileIcon/>,
                tooltip: 'Файл',
                handle: this.showChoseFileDialog.bind(this, 'media'), isActive: true,
            },
            {
                title: <PdfIcon/>,
                tooltip: 'PDF',
                handle: this.showChoseFileDialog.bind(this, 'pdf'), isActive: true,
            },
            {
                title: <FineIcon/>,
                tooltip: 'Штраф',
                handle: this.showFineDialog.bind(this, true), isActive: true,
            },
        ];
    }

    getSecondLineControls() {
        const SECOND_LINE_CONTROLS: any = [];
        const forwardControl = this.state.selectedMessages.length
            ? {
                title: <span>Переслать в GO ({this.state.selectedMessages.length})</span>,
                tooltip: 'Переслать сообщения',
                handle: this.showForwardWindow.bind(this, true),
                isActive: true,
            }
            : null;
        const cancelSelectControl = this.state.selectedMessages.length
            ? {
                title: <span>Сбросить</span>,
                tooltip: 'Отменить выделение сообщений',
                handle: this.cancelSelect.bind(this),
                isActive: true,
            }
            : null;

        forwardControl ? SECOND_LINE_CONTROLS.push(forwardControl) : null;
        cancelSelectControl ? SECOND_LINE_CONTROLS.push(cancelSelectControl) : null;

        return SECOND_LINE_CONTROLS;
    }

    getOriginatorName(): string {
        const userId = this.props.selectedChatItem?.originator;
        const userInfo = this.props.users?.find(user => user.id === userId) ?? {};

        return UserInfoHandler.getName.call(userInfo) ?? '';
    }

    render() {
        const initText = this.setTextFromStorage();
        const additionalControls: IRichEditorControlItem[] = this.getAdditionalControls();
        let fastAnswers: any;
        try {
            fastAnswers = this.props.settings.fastAnswers;
        } catch (err) {
            fastAnswers = [];
        }

        const chatIsClosed = this.state.selectedChatData.expected_action?.type === ExpectedActionType.chat_closed;
        const SECOND_LINE_CONTROLS = this.getSecondLineControls();
        const showSendAndHide = this.props.selectedChatItem?.tag_data?.external_chat?.external_chat_id;
        const currentUserId = this.props.currentUserId;
        const performerId = this.props.selectedChatItem?.tag_data?.performer;
        const pdfUploading = this.state.fileWindowState === 'pdf';
        const fileUploadMethod = pdfUploading ? this.uploadPdf.bind(this) : this.uploadFiles.bind(this);

        return <div className={`${chatStyle.chats_column} ${style.component}`}>
            {
                this.state.isEmpty
                    ? null
                    : <>
                        <ChatWindowHeader getImgLinks={this.getImgLinks()}
                                          updateCategorizations={this.props.getClassification
                                              ?.bind(this, this.getTagId())}
                                          categorizations={this.props.classificationData}
                                          changeMode={this.changeMode.bind(this)}/>
                        {
                            this.state.isLoading
                                ? <Spin/>
                                : <div className={`${style.chat_window_wrap} ${chatIsClosed ? style.closed_chat : ''}`}>
                                    {this.state.error ? <SimpleError error={this.state.error}/> : null}

                                    <ChatWindowContent selectedChatData={this.state.selectedChatData}
                                                       tagData={this.props.selectedChatItem?.tag_data}
                                                       categorizations={this.props.classificationData}
                                                       isSafeMode={this.state.isSafeMode}
                                                       selectedMessages={this.state.selectedMessages}
                                                       changeSelectedMessages={this.changeSelectedMessages.bind(this)}/>
                                    {!chatIsClosed && currentUserId === performerId
                                        ? <RichEditor onChange={this.onRichEditorChange.bind(this)}
                                                      initData={initText || this.state.initText}
                                                      fastAnswers={fastAnswers}
                                                      tag_id={this.getTagId()}
                                                      originatorName={this.getOriginatorName()}
                                                      additionalControls={additionalControls}
                                                      ref={this.richEditor}
                                                      className={`${style.editor}`}
                                                      secondLineControls={SECOND_LINE_CONTROLS}
                                                      onSendAndHide={showSendAndHide
                                                          ? this.onSendAndHide.bind(this)
                                                          : null}
                                                      onSendText={this.onSendText.bind(this, {})}/>
                                        : !chatIsClosed
                                            ? <div className={style.perform_warning}>
                                                Возьмите чат в работу, чтобы сюда написать
                                            </div>
                                            : null
                                    }
                                </div>
                        }
                        {
                            this.state.showBonusDialog
                            && <BonusControlDialog onClose={this.showBonusDialog.bind(this, false)}
                                /* sendMessage={
                                     this.onSendText.bind(this,
                                         {
                                             statusText: 'Отправка бонуса',
                                             message_type: MESSAGE_TYPE.BONUS,
                                             customHandler: true
                                         }
                                     )}*/
                                                   chat_id={this.props.selectedChatItem?.id}
                                                   userId={this.props.selectedChatItem
                                                   && this.props.selectedChatItem.originator || ''}
                                                   from={[chatLink(this.props.selectedChatItem)]}/>
                        }
                        {
                            this.state.showSessionDialog
                            && <SessionDialog onClose={this.showSessionDialog.bind(this, false)}
                                              sendMessage={
                                                  this.onSendText.bind(this,
                                                      {
                                                          statusText: 'Отправка сессии',
                                                          message_type: MESSAGE_TYPE.ORDER,
                                                          customHandler: true,
                                                      })
                                              }
                                              selectedChat={this.props.selectedChatItem}/>
                        }

                        {
                            this.state.showStickerDialog
                            && <StickerDialog onClose={this.showStickerDialog.bind(this, false)}
                                              sendMessage={
                                                  this.onSendText.bind(this,
                                                      {
                                                          statusText: 'Отправка стикера',
                                                          message_type: MESSAGE_TYPE.STICKER,
                                                          customHandler: true,
                                                      })
                                              }
                                              selectedChat={this.props.selectedChatItem}/>
                        }

                        {
                            this.state.fileWindowState
                            && <FileChooserDialog onClose={this.showChoseFileDialog.bind(this, false)}
                                                  upload={fileUploadMethod}
                                                  isUploading={this.state.isUploading}
                                                  error={this.state.filesError}
                                                  onlyPDF={pdfUploading}
                                                  fileCountLimit={FILE_COUNT_LIMIT}/>
                        }

                        {
                            this.state.showFineDialog
                            && <FineDialogModal onClose={this.showFineDialog.bind(this, false)}
                                                userId={this.props?.selectedChatItem?.originator || ''}
                                                sendMessage={
                                                    this.onSendText.bind(this,
                                                        {
                                                            statusText: 'Отправка штрафа',
                                                            message_type: MESSAGE_TYPE.FINE,
                                                            customHandler: true,
                                                        })
                                                }/>
                        }
                        {this.state.showForwardWindow
                            ? <ForwardWindow showForwardWindow={this.showForwardWindow.bind(this)}
                                             user_id={this.props.selectedChatItem?.originator ?? ''}
                                             chat_id={this.props.selectedChatItem?.tag_data?.topic_link ?? ''}
                                             cancelSelect={this.cancelSelect.bind(this)}
                                             messages={this.state.selectedMessages}/>
                            : null
                        }

                        {this.state.showClassificationModal
                            ? <ClassificationDialog onClose={this.closeClassificationModal.bind(this)}
                                                    onClassified={this.onClassified.bind(this)}
                                                    tag_id={this.getTagId() ?? null}
                                                    type={ClassificationType.chat}
                                                    user_id={this.props.selectedChatItem?.originator ?? null}
                                                    chat_id={this.props.selectedChatItem?.id ?? null}/>
                            : null
                        }
                    </>
            }
            {this.state.validationConfirmOpen
                ? <Confirm question={VALIDATION_CONFIRM_QUESTION}
                           onClose={this.openValidationConfirm.bind(this, false)}
                           accept={this.acceptValidationConfirm.bind(this)}
                           error={null}/>
                : null
            }
        </div>;
    }
}
