import * as React from 'react';

import { GetClassification, IGetClassificationExport } from '../../../../../decorators/GetClassification';
import { Button, ButtonTypes } from '../../../../ui/Button';
import { ErrorsModal } from '../../../../ui/ErrorBar';
import { Window } from '../../../../ui/FullModal';
import { Link } from '../../../../ui/Link';
import Spin from '../../../Spin';
import { ClassificationList } from '../ClassificationUtils/ClassificationList';
import { History } from '../ClassificationUtils/History';
import { ClassificationDialogContext } from '../context';
import style from '../index.css';
import { IClassificationModule } from '../types';

interface IClassificationDialogWrapperProps extends IGetClassificationExport {
    onClassified: Function;
    classificationError?: Error | null;
    onClose: Function;
    setClassification: (afterClassificationAction: Function | null) => void;
    showCoreComponentsOnly: boolean;
}

interface IClassificationDialogWrapperState {
    openErrorModal: boolean;
    openHistoryModal: boolean;
}

@GetClassification()
export class ClassificationDialogWrapper extends
    React.Component<IClassificationDialogWrapperProps & IClassificationModule, IClassificationDialogWrapperState> {
    state: IClassificationDialogWrapperState = {
        openErrorModal: false,
        openHistoryModal: false,
    };
    static contextType = ClassificationDialogContext;
    context!: React.ContextType<typeof ClassificationDialogContext>;

    getData() {
        this.context.tag_id && this.props.getClassification?.(this.context.tag_id);
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const {
            sentClassification,
            sendingClassification,
            category,
            tag_id,
            sendingClassificationError,
        } = this.context;
        const { children, onClose, classificationError, showCoreComponentsOnly } = this.props;
        const { openErrorModal, openHistoryModal } = this.state;

        return <>
            <Window onClose={onClose?.bind(this)}
                    title={`Классификация`}
                    className={style.classification_window}
                    error={classificationError}>
                <div className={`${style.component}`}>
                    <div className={`${style.section} ${style.left}`}>
                        <div className={style.historyBtn}
                             onClick={() => {
                                 this.setState((prev) => ({
                                     openHistoryModal: !prev.openHistoryModal,
                                 }));
                             }}>
                            история {openHistoryModal ? ' <' : ' >'}
                        </div>

                        <div className={style.children}>{children}</div>

                        <div className={sendingClassificationError ? style.errors : ''}>
                            {sendingClassificationError
                                ? <Link onClick={() => {
                                    this.setState({
                                        openErrorModal: true,
                                    });
                                }}
                                        className={style.error_link}>
                                    Упс, ошибочка :( Подробности по клику
                                </Link>
                                : null
                            }
                        </div>
                        <div className={sentClassification ? style.success : ''}>
                            {sentClassification
                                ? <div className={style.success_label}>Классификация добавлена успешно</div>
                                : null
                            }
                        </div>
                        <div className={style.add_buttons}>
                            {!showCoreComponentsOnly && <Button onClick={() => {
                                this.props.setParameters({ addAnotherClassification: true });
                                this.props.setClassification(this.getData.bind(this));
                            }}
                                                                isLoading={sendingClassification}
                                                                disabled={!category}
                                                                colorType={ButtonTypes.warning}>
                                ОК и добавить новую
                            </Button>}
                            <Button onClick={this.props.setClassification.bind(this, null)}
                                    isLoading={sendingClassification}
                                    disabled={!category}
                                    colorType={ButtonTypes.positive}>
                                ОК
                            </Button>
                        </div>
                        <Button onClick={this.getData.bind(this)}
                                disabled={sendingClassification}>
                            Обновить список
                        </Button>

                        {this.props.classificationIsLoading
                            ? <Spin/>
                            : <ClassificationList tag_id={tag_id}
                                                  onClassified={this.props.onClassified.bind(this)}
                                                  classificationData={this.props.classificationData}/>
                        }

                    </div>

                    {openHistoryModal
                        ? <History tag_id={tag_id}/>
                        : null
                    }
                </div>
            </Window>
            {openErrorModal
                ? <ErrorsModal errors={[sendingClassificationError]}
                               onClose={() => {
                                   this.setState({
                                       openErrorModal: false,
                                   });
                               }}
                               title={'Просмотр ошибки'}/>
                : null
            }
        </>;
    }
}
