import moment from 'moment';
import React from 'react';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { TagRecordHandler } from '../../../../models/tag';
import { UserInfoHandler } from '../../../../models/user';
import { FormatDateInString } from '../../../../ui/FormatDate';
import { Request2 } from '../../../../utils/request';
import { parseUA } from '../../../../utils/utils';
import { CLASSIFICATION_REQUESTS, REQUESTS } from '../requests';

interface ISendCommentProps {
    user_id: string | null;
    tag_id: string;
    chat_id: string;
    task_id: string;
    session_id: string | undefined | 0;
    comment: string;

    setError: (error: Error) => void;
}

export interface ITaskSenderExport {
    sendComment?: (props: ISendCommentProps, callback?: Function) => void;
}

export const TaskSender = () => (Component): typeof Component => {
    return class extends React.Component<any, any> {
        state = {
            stopRequest: false,
        };
        request = new Request2({ requestConfigs: CLASSIFICATION_REQUESTS });

        sendComment(props: ISendCommentProps, callback) {
            this.getData(props)
                .then(() => {
                    this.setState({
                        stopRequest: false,
                    });
                    callback();
                });
        }

        async getData(props: ISendCommentProps) {
            const { user_id, tag_id, chat_id, task_id, session_id, comment, setError } = props;
            const { device, name } = await this.getUserData(user_id, setError);
            const { date, tag } = await this.getTagDetails(tag_id, setError);
            const {
                requestLink,
                userLink,
                sessionLink,
                tagName,
            } = this.generateLinks(user_id, chat_id, tag_id, session_id, tag);
            const short_task_id = this.getTaskId(task_id);

            const startText = '**Получили обращение с симптомами, похожими на задачу.**';
            const dateDesc = 'Дата: ';
            const requestDesc = 'Ссылка на обращение';
            const deviceDesc = 'Девайс: ';
            const sessionDesc = 'Привязанная сессия';
            const commentDesc = 'Комментарий: ';
            const tagNameDesc = 'Тег коммуникации: ';

            const finalText = startText + `\n`
                + dateDesc + (date || EMPTY_DATA) + `\n`
                + tagNameDesc + (tagName || EMPTY_DATA) + `\n`
                + (requestLink ? `((${requestLink} ${requestDesc}))` : EMPTY_DATA) + `\n`
                + (userLink ? `((${userLink} ${name}))` : EMPTY_DATA) + `\n`
                + deviceDesc + (device || EMPTY_DATA) + `\n`
                + (sessionLink ? `((${sessionLink} ${sessionDesc}))` : `${sessionDesc}: ${EMPTY_DATA}`) + `\n`
                + commentDesc + (comment || EMPTY_DATA);

            if (!this.state.stopRequest) {
                await this.request.exec(REQUESTS.ADD_COMMENT, {
                    queryParams: {
                        task_id: short_task_id,
                    },
                    body: {
                        text: finalText,
                    },
                })
                    .catch(error => {
                        setError(error);
                    });
            }
        }

        getUserData(user_id: string | null, setError): any {
            if (user_id) {
                return this.request.exec(REQUESTS.GET_USER_INFO, {
                    queryParams: { user_id },
                })
                    .then(response => {
                        const device_info: any = response?.realtime_data?.user_agent
                            ? parseUA(response?.realtime_data?.user_agent)
                            : {};

                        const OSV = device_info.device;
                        const AppV = response?.realtime_data?.app_version;
                        const device = `${OSV ?? ''}${OSV && AppV ? ', App:' : ''}${AppV ?? ''}`;

                        const name = UserInfoHandler.getPrintNameWithoutUserName.call(response);

                        return { device, name } ?? {};
                    })
                    .catch(error => {
                        this.setState({ stopRequest: true });
                        setError(error);
                    });
            }

            this.setState({ stopRequest: true });

            return {};

        }

        getTagDetails(tag_id: string, setError): any {
            if (tag_id) {
                const monthBefore = +moment(new Date()).subtract(1, 'year');
                const since = +((monthBefore / ONE_SECOND).toFixed(0));

                return this.request.exec(REQUESTS.GET_TAG_DETAILS, {
                    queryParams: { tag_id, since },
                })
                    .then(response => {
                        const records = response?.records ?? [];
                        const firstRec = records[records?.length - 1];

                        return {
                            date: FormatDateInString({ value: firstRec?.timestamp * ONE_SECOND }),
                            tag: firstRec,
                        } ?? {};
                    })
                    .catch(error => {
                        this.setState({ stopRequest: true });
                        setError(error);
                    });
            }

            this.setState({ stopRequest: true });

            return {};

        }

        generateLinks(user_id, chat_id, tag_id, session_id, tag) {
            let requestLink;
            const admin_href = 'https://carsharing.yandex-team.ru/#';
            const tagName = TagRecordHandler.getTagName.call(tag);
            const topic_link = tag.topic_link || tag.tag_details.topic_link;

            if (topic_link && user_id && chat_id && tag_id) {
                if (topic_link.includes('outgoing')) {
                    requestLink = `${admin_href}/outgoing?user_id=${user_id}&chat_id=${chat_id}&tag_id=${tag_id}`;
                } else {
                    requestLink = `${admin_href}/chats?user_id=${user_id}&chat_id=${chat_id}&tag_id=${tag_id}`;
                }
            }

            const userLink = user_id ? `${admin_href}/clients/${user_id}` : null;
            const sessionLink = session_id ? `${admin_href}/session/${session_id}` : null;

            return { requestLink, userLink, sessionLink, tagName };
        }

        getTaskId(task: string): string | null {
            if (task) {
                const ST_ID = 'st.yandex-team.ru/';
                const id = task.substr(task.indexOf(ST_ID) + ST_ID.length);

                return id.trim();
            }

            this.setState({ stopRequest: true });

            return null;

        }

        render() {
            return <Component {...this.props}
                              sendComment={this.sendComment.bind(this)}/>;
        }
    };
};
