import * as React from 'react';

import StIcon from '../../../../../svg-components/st.component.svg';
import { ClassificationType } from '../../../../../types';
import { EMPTY_DATA } from '../../../../constants';
import Select, { IOption, IOptionInfo } from '../../../../ui/Select';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { ITreeData } from '../../types';
import { CLASSIFICATION_REQUESTS, REQUESTS } from '../requests';
import * as style from './index.css';

export interface ITreeOption {
    id: string;
    meta?: {
        additional_data: ITreeOptionAdditional;
        call_direction: string;
        enabled: boolean;
        label: string;
        order: number;
        origins: string[];
    };
}

export interface ITreeOptionAdditional {
    st_links?: IClassificationMetaItem[];
}

export interface IClassificationMetaItem {
    hint: string;
    link: string;
    type: string;
}

interface ITreeSuggestProps {
    onSelect: (id: any) => void;
    setOptions: (data: any) => void;
    initialValues: any;
    type: ClassificationType;
    keyword?: string;
}

interface ITreeSuggestState {
    isLoading: boolean;
    data: ITreeData | null;
    error: Error | null;
    options: IOptionInfo[];
    formattedData: any;
    selectedData: any;
}

export class TreeSuggest extends React.Component<ITreeSuggestProps, ITreeSuggestState> {
    state: ITreeSuggestState = {
        isLoading: false,
        data: null,
        error: null,
        options: [],
        formattedData: {},
        selectedData: null,
    };
    request = new Request2({ requestConfigs: CLASSIFICATION_REQUESTS });

    componentDidMount(): void {
        this.getTree(true);
    }

    componentDidUpdate(prevProps: Readonly<ITreeSuggestProps>, prevState: Readonly<ITreeSuggestState>, snapshot?: any) {
        const selectedDataChanged = !isObjectEqual(this.state.selectedData, prevState.selectedData);
        const initialValuesChanged = this.props.initialValues !== prevProps.initialValues;

        if (!selectedDataChanged && initialValuesChanged) {
            this.onSelect(this.props.initialValues);
        }
    }

    onSelect(id: string) {
        const selectedData: ITreeOption = id ? this.state.formattedData[id] ?? { id } : '';
        this.setState({ selectedData });
        this.props.onSelect(selectedData);
    }

    getTreeOptions(data: ITreeData[], opt: { index?: string; prev?: string[] } = {}) {
        const _data = [...data];
        const keyword = this.props?.keyword;

        _data.sort((a, b) => {
            return a?.meta?.order - b?.meta?.order;
        });

        const path = opt?.prev?.reverse() || [];

        return _data.reduce((result: IOptionInfo[], item, index: number) => {
            const indexLabel = opt.index ? `${opt.index}.${++index}` : `${++index}`;
            const value = item.id;
            const name = item?.meta?.label || EMPTY_DATA;

            const pathJoiner = '➔';
            const pathRender = path.map((el, index) => {
                return <span key={index}><span className={style.prev_path_item}>{el}</span>
                    <span className={style.prev_path_joiner}>{pathJoiner}</span>
                </span>;
            });
            pathRender.push(<span className={style.prev_path_name} key={pathRender.length}>{name}</span>);

            const description = `${item?.meta?.keywords?.join(', ') || ''}`;

            const isChatOrigins = item?.meta?.origins?.some(el => el.includes(this.props.type));
            const isIncludesKeyWord = keyword && item?.meta?.keywords?.includes(keyword);
            const enabled = item?.meta?.enabled;
            const hasST = this.isItemHavSt(item);

            if (!enabled) {
                return result;
            }

            const text = <div className={`${style.tree_suggest_item} ${opt.index ? style.children : ''}`}
                              key={indexLabel}>
                <div className={style.item_title}>{name}
                    <span className={style.prev_path}>{pathRender}</span>
                </div>

                {hasST
                    ? <StIcon className={style.st_icon} key={name}/>
                    : null
                }
            </div>;

            if (!item.children) {
                if (keyword) {
                    if (isIncludesKeyWord && isChatOrigins) {
                        result.push({ value, text, description, selectedDisplayText: name });
                    }
                } else if (isChatOrigins) {
                    result.push({ value, text, description, selectedDisplayText: name });
                }
            }

            if (item.children && item.children.length) {
                result.push(...this.getTreeOptions(item.children, {
                    index: indexLabel,
                    prev: [name, ...opt?.prev || []],
                }));
            }

            return result;
        }, []);
    }

    getTree(withLoader?: boolean) {
        this.setState({
            isLoading: !!withLoader,
        }, () => {
            this.request.exec(REQUESTS.GET_CLASSIFICATION_TREE)
                .then(response => {
                    this.setState({
                        isLoading: false,
                        error: null,
                        data: response?.tree || {},
                        formattedData: this.formatData(response?.tree?.children),
                    }, () => {
                        this.setState({
                            options: this.getTreeOptions(response?.tree?.children),
                        }, () => this.getOptionsToParent());
                    });
                })
                .catch(err => {
                    this.setState({
                        isLoading: false,
                        error: err,
                        data: null,
                    });
                });
        });
    }

    getOptionsToParent() {
        this.props.setOptions(this.state.options);
    }

    isItemHavSt(item) {
        return this.state.formattedData[item.id]?.meta?.additional_data?.st_links?.length;
    }

    formatData(data): any {
        const formattedData = {};

        function addValue(item) {
            if (!formattedData[item.id]) {
                formattedData[item.id] = item;
            }

            if (item.children) {
                item.children.forEach(el => addValue(el));
            }
        }

        data.forEach(item => addValue(item));

        return formattedData;
    }

    sortWhenSearching(options: IOption[], searchValue: string) {
        return options?.sort((option1, option2) => {
            const option1Name = option1.info.selectedDisplayText;
            const option2Name = option2.info.selectedDisplayText;

            return option1Name && option2Name
                ? +(option2Name.toLowerCase().includes(searchValue.toLowerCase()))
                - +(option1Name.toLowerCase().includes(searchValue.toLowerCase()))
                : 0;
        });
    }

    render() {
        const { initialValues } = this.props;
        const { options, error, isLoading } = this.state;
        if (error) {
            return <SimpleError error={error} data={{ label: 'Ошибка при загрузке дерева' }}/>;
        }

        if (isLoading) {
            return <div className={style.tree_suggest}>
                <Spin/>
            </div>;
        }

        return <Select options={options}
                       containerClassName={style.select}
                       sortWhenSearching={this.sortWhenSearching.bind(this)}
                       initialValues={initialValues}
                       onSelect={this.onSelect.bind(this)}
                       placeholder={'Фильтр/Дерево'}/>;
    }
}
