import React from 'react';

import TreeIcon from '../../../../svg-components/tree.component.svg';
import { ClassificationType } from '../../../../types';
import { Collapse2 } from '../../../ui/Collapse2';
import { Window } from '../../../ui/FullModal';
import TextArea from '../../../ui/TextArea';
import { Request2 } from '../../../utils/request';
import ClassificationTree from '../../Settings/ClassificationTree/component';
import ClassificationCloudSuggest from './ClassificationCloudSuggest';
import ClassificationComment from './ClassificationComment';
import ClassificationDialogWrapper from './ClassificationDialogWrapper';
import ClassificationMeta from './ClassificationMeta';
import ClassificationST from './ClassificationST';
import { ClassificationDialogContext } from './context';
import style from './index.css';
import { CLASSIFICATION_REQUESTS, REQUESTS } from './requests';
import { IClassificationMetaItem, ITreeOption, TreeSuggest } from './TreeSuggest';
import {
    ClassificationDialogParametersKeys,
    IClassificationDialogParameters,
    IClassificationQueryBody,
} from './types';

interface IClassificationDialogProps {
    tag_id: string | null;
    onClose: Function;
    onClassified: Function;
    type: ClassificationType;
    user_id: string | null;
    chat_id: string | null;
    suggest?: boolean;
    keyword?: string;
    showCoreComponentsOnly?: boolean;
    blockRules?: Record<string, boolean>;
}

type TClassificationState = Omit<IClassificationDialogParameters, keyof IClassificationDialogProps>;
export type TSetParameters = Partial<TClassificationState>;

export class ClassificationDialog extends React.Component<IClassificationDialogProps, TClassificationState> {
    state: TClassificationState = {
        category: null,
        comment: '',
        stData: null,
        session_id: undefined,
        selectedST: null,
        sentClassification: false,
        sendingClassification: false,
        sendingSTMessage: false,
        sentSTMessage: false,
        coordinates: null,
        scoring: [],
        violation: [],
        addAnotherClassification: false,
        sendingClassificationError: null,
        isClassificationTreeOpened: false,
    };

    request = new Request2({ requestConfigs: CLASSIFICATION_REQUESTS });
    textarea: React.RefObject<TextArea>;

    componentDidMount() {
        this.textarea = React.createRef();
    }

    componentWillUnmount() {
        this.request.abort();
    }

    componentDidUpdate(prevProps, prevState) {
        if (this.state.sentSTMessage !== prevState.sentSTMessage && this.state.sentSTMessage) {
            this.finishSettingClassification();
        }
    }

    setClassification(afterClassificationAction: Function | null) {
        this.setState({
            sendingClassification: true,
            sentClassification: false,
        }, () => {
            const body = this.getQueryBody();

            if (body && body.meta_info?.session_id !== null) {
                this.request.exec(REQUESTS.SET_CLASSIFICATION, { body })
                    .then(() => {
                        afterClassificationAction?.();
                        if (!this.state.selectedST) {
                            this.finishSettingClassification();
                        } else {
                            this.setState({
                                sendingSTMessage: true,
                                sentClassification: true,
                                sendingClassification: false,
                            });
                        }
                    })
                    .catch(error => {
                        this.setState({
                            sendingClassificationError: error,
                            sendingClassification: false,
                            sentClassification: false,
                        });
                    });
            } else {
                this.setState({
                    sendingClassificationError: new Error(`Нет ${body?.meta_info?.session_id === null
                        ? 'session_id' : 'tag_id'}`),
                    sendingClassification: false,
                    sentClassification: false,
                });
            }
        });
    }

    finishSettingClassification() {
        this.setState({
            category: null,
            stData: null,
            selectedST: null,
            sentSTMessage: false,
            sendingSTMessage: false,
            sentClassification: true,
            sendingClassification: false,
        }, () => {
            if (!this.state.addAnotherClassification) {
                this.props.onClassified();
            } else {
                this.setState({
                    addAnotherClassification: false,
                });
            }
        });
    }

    getQueryBody(): IClassificationQueryBody | undefined {
        if (this.props.tag_id) {
            const body: IClassificationQueryBody = {
                tag_id: this.props.tag_id,
                comment: this.state.comment,
                category: this.state.category ?? '',
                meta_info: {
                    session_id: this.state.session_id,
                },
            };

            this.state.coordinates ? body.meta_info.coordinates = this.state.coordinates : null;
            this.state.scoring.length ? body.meta_info.scoring = this.state.scoring : null;
            this.state.violation.length ? body.meta_info.violation = this.state.violation : null;

            if (this.state.selectedST) {
                const item: IClassificationMetaItem | undefined = this.state.stData?.additional_data?.st_links
                    ?.find(el => el.link === this.state.selectedST);

                item ? body.meta_info.st = item : null;
            }

            return body;
        }

        return;
    }

    setParameters(dict: TSetParameters) {
        this.setState((prev) => ({ ...prev, ...dict }), () => {
            const keys = Object.keys(dict);
            for (const parameter in keys) {
                if (parameter === ClassificationDialogParametersKeys.category) {
                    this.textarea?.current?.focus();
                }
            }
        });
    }

    onSelectOption(value: ITreeOption) {
        this.setParameters({ [ClassificationDialogParametersKeys.category]: value?.id });
        this.setParameters({ [ClassificationDialogParametersKeys.stData]: value?.meta });
    }

    setClassificationFromModal(value: ITreeOption) {
        this.onSelectOption(value);
        this.openClassificationTree(false);
    }

    openClassificationTree(isClassificationTreeOpened: boolean) {
        this.setState({ isClassificationTreeOpened });
    }

    render() {
        const {
            category,
            sentClassification,
            isClassificationTreeOpened,
        } = this.state;

        const {
            onClose,
            suggest,
            keyword,
            type,
            user_id,
            tag_id,
            chat_id,
            showCoreComponentsOnly,
            blockRules,
        } = this.props;

        const showCloud = blockRules?.ml_classification;

        return <ClassificationDialogContext.Provider value={{
            ...this.state,
            user_id,
            tag_id,
            chat_id,
        }}>
            <ClassificationDialogWrapper setParameters={this.setParameters.bind(this)}
                                         showCoreComponentsOnly={showCoreComponentsOnly ?? false}
                                         onClassified={this.props.onClassified}
                                         setClassification={this.setClassification.bind(this)}
                                         onClose={onClose}>

                {showCloud
                    ? <ClassificationCloudSuggest suggest={suggest ?? true}
                                                  setParameters={this.setParameters.bind(this)}/>
                    : null
                }

                <div className={style.tree_suggest_wrapper}>
                    <TreeSuggest setOptions={() => {}}
                                 onSelect={this.onSelectOption.bind(this)}
                                 keyword={keyword}
                                 initialValues={category}
                                 type={type}/>
                    <TreeIcon className={style.tree_button}
                              onClick={this.openClassificationTree.bind(this, true)}/>
                </div>

                <ClassificationComment setParameters={this.setParameters.bind(this)}
                                       textareaRef={this.textarea}/>

                <Collapse2 headerClassname={style.collapse}
                           title={'Все поля классификации'}
                           initialExpanded={!showCoreComponentsOnly}>
                    <ClassificationST setParameters={this.setParameters.bind(this)}
                                      shouldSendComment={sentClassification}/>

                    <ClassificationMeta setParameters={this.setParameters.bind(this)}/>
                </Collapse2>
            </ClassificationDialogWrapper>

            {isClassificationTreeOpened
                ? <Window onClose={this.openClassificationTree.bind(this, false)}
                          className={style.classification_tree}
                          title={'Дерево классификаций'}>
                    <ClassificationTree readOnly={true}
                                        type={type}
                                        onSelect={this.setClassificationFromModal.bind(this)}/>
                </Window>
                : null
            }

        </ClassificationDialogContext.Provider>;
    }
}
