import React from 'react';

import { Button, ButtonTypes } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { Request2 } from '../../../utils/request';
import { buttonNameDetails } from '../../../utils/sendLogs/eventTypes/buttonDetails';
import { IStore } from '../../App/store';
import { FeedList } from '../../ChatsCommon/FeedList';
import FeedListItem from '../../ChatsCommon/FeedListItem/index';
import * as commonChatStyle from '../../ChatsCommon/index.css';
import { _CHATS } from '../../ChatsCommon/types';
import QueryScheduler, { IProgressData } from '../../QueryScheduler/QueryScheduler';
import { ChatToggle } from '../ChatToggle';
import { COUNT_FIRST } from '../constants';
import { CHAT_REQUESTS, REQUESTS } from '../request';
import { FeedStateToProps } from './index';

interface IFeedProps extends IStore, FeedStateToProps {
    toggle: () => void;
    feedIsOpen: boolean;
    setSelectList?: (type: _CHATS, menu: any, chat?: any) => void;
    selectChatItem?: (chatItem: Record<string, any>) => void;
}

interface IFeedState {
    filter: string;
    performNChatsIsOpen: boolean;
    isLoading: boolean;
    progress: any;
}

export class Feed extends React.Component<IFeedProps, IFeedState> {
    state: IFeedState = {
        filter: '',
        performNChatsIsOpen: false,
        isLoading: false,
        progress: null,
    };
    request = new Request2({ requestConfigs: CHAT_REQUESTS });

    onChange(filter) {
        this.setState({
            filter,
        });
    }

    showPerformNChats(performNChatsIsOpen) {
        this.setState({
            performNChatsIsOpen,
        });
    }

    perform() {
        const { feedData, countToPerformChats } = this.props;
        const queue: Promise<any> [] = [];
        const chatToPick = countToPerformChats === 1 ? feedData?.slice(0, countToPerformChats)?.[0] : null;
        const chats = feedData?.slice(0, countToPerformChats).map(el => el?.tag_data?.tag_id);

        chats.forEach(tag_id => {
            queue.push(this.request.exec.bind(this.request, REQUESTS.SET_PERFORM, { body: { tag_id } }));
        });

        const qs = new QueryScheduler({
            queue,
            limit: COUNT_FIRST,
            onProgress: this.onProgress.bind(this),
            onSuccess: this.onSuccess.bind(this, chatToPick),
        });

        chats.length && this.setState({ isLoading: true }, () => {
            qs.run();
        });
    }

    onProgress(data: IProgressData) {
        if (data.failed?.length) {
            const progress = `${data.failed?.length}/${data.queue?.length}`;
            this.setState(() => ({
                progress: 'Не получилось взять чаты в работу ' + progress,
            }));
        }
    }

    onSuccess(chat, data: IProgressData) {
        const { whoAmI, setSelectList } = this.props;

        if (chat && chat.tag_data) {
            chat.tag_data.performer = whoAmI;
        }

        this.setState({
            isLoading: false,
        }, () => {
            if (!data.failed?.length) {
                setSelectList?.(_CHATS.MY, chat?.tag_data?.tag ?? '', chat);
            }
        });
    }

    resetProgress() {
        this.setState({
            progress: null,
        });
    }

    shouldComponentUpdate(nextProps: Readonly<IFeedProps>, nextState: Readonly<IFeedState>, nextContext: any): boolean {
        return !isObjectEqual(this.props, nextProps) || !isObjectEqual(this.state, nextState);
    }

    render() {
        let {
            feedData,
            feedIsOpen,
            toggle,
            selectedMenu,
            users,
            isLoading,
        } = this.props;
        const { filter, progress } = this.state;
        const isNewChats = selectedMenu === _CHATS.NEW;
        const menuForShow = selectedMenu === _CHATS.MY || selectedMenu === _CHATS.CUSTOM;
        const hideFeed = !this.props.blockRules.SupportShowAllChats && !menuForShow;

        if (filter) {
            feedData = feedData?.filter(el => {
                return el?.last_message?.text?.toLocaleUpperCase()?.includes(filter.toLocaleUpperCase());
            }) || [];
        }

        return <div className={`${commonChatStyle.common_feed} ${commonChatStyle.chat_feed}`}>
            <div className={commonChatStyle.controls}>
                <ChatToggle isOpen={feedIsOpen} toggle={toggle.bind(this)}/>

                {isNewChats && feedIsOpen
                    ? <Button onClick={this.perform.bind(this)}
                              colorType={ButtonTypes.positive}
                              isLoading={this.state.isLoading}
                              ytLog={{ button_name: buttonNameDetails.PERFORM_N_CHATS }}>
                        Забрать чат
                    </Button>
                    : null
                }

                {progress
                    ? <ProgressModal progress={progress}
                                     onClose={this.resetProgress.bind(this)}/>
                    : null
                }

                {feedIsOpen && selectedMenu && !hideFeed
                    ? <Input value={filter}
                             onChange={this.onChange.bind(this)}
                             placeholder={'фильтр'}/>
                    : null
                }
            </div>

            {feedIsOpen && !hideFeed
                ? <FeedList data={feedData}
                            users={users}
                            isLoading={isLoading}
                            markOutgoing={true}
                            feedListItem={FeedListItem}/>
                : null
            }
        </div>;
    }
}

interface IPerformNChatsProps {
    progress: any;
    onClose: () => void;
}

export const ProgressModal = React.memo((props: IPerformNChatsProps) => {
    return <Window onClose={props.onClose.bind(null)} title={`Не удалось взять в работу`}>
        {props.progress}
    </Window>;
});
