import { RouterState } from 'connected-react-router';
import React from 'react';

import { Confirm } from '../../../ui/FullModal';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { NAVIGATION } from '../../../utils/navigation';
import { Request2 } from '../../../utils/request';
import { ytLogs } from '../../../utils/sendLogs';
import { EVENT_TYPES } from '../../../utils/sendLogs/eventTypes/eventTypes';
import { IStore } from '../../App/store';
import { ARCHIVE_TITLE, DEFER_TITLE } from '../../ChatsCommon/constants';
import * as chatStyle from '../../ChatsCommon/index.css';
import { LinesList } from '../../ChatsCommon/LinesList/component';
import { _CHATS, IAdditionalMenu, LinesMenuType, NOTIFY_PERMISSION } from '../../ChatsCommon/types';
import { FAV_ACTION, FAV_LINES_TYPES, FAVOURITES, FavouritesLines, IFavLineItem } from '../../FavouritesLines';
import { ChatToggle } from '../ChatToggle';
import { CHAT_REQUESTS, REQUESTS } from '../request';
import { LinesListDispatchToProps } from './index';
import * as style from './index.css';

interface ILinesListProps extends IStore, LinesListDispatchToProps, RouterState {
    lineIsOpen: boolean;
    getHistory: (user: any, type: _CHATS) => void;
    toggle: () => Function;
    linesListData: LinesMenuType;
    selectedMenu: string;
    selectedMenuItem: string;
}

export class LinesListComponent extends React.Component<ILinesListProps> {
    state = {
        isAddConfirmOpen: false,
        isAddAction: true,
        group: '',
    };
    chosenItem: Record<string,any> | null = null;
    fvLines = new FavouritesLines();
    request = new Request2({ requestConfigs: CHAT_REQUESTS });
    log = ytLogs.getInstance();

    shouldComponentUpdate(nextProps: Readonly<ILinesListProps>, nextState: Readonly<{}>, nextContext: any): boolean {
        return !isObjectEqual(nextProps, this.props) || !isObjectEqual(nextState, this.state);
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    selectMenu(type, menu) {
        location.hash = NAVIGATION.ROUTES.CHATS.uri;
        this.props.setSelectList(type, menu);
    }

    selectMenuDeffer() {
        const { setFeedIsLoading, setSelectList, setChats } = this.props;

        location.hash = NAVIGATION.ROUTES.CHATS.uri;
        this.request.abort();
        setFeedIsLoading(true);
        setSelectList(_CHATS.DEFFER, null);
        this.request.exec(REQUESTS.FEED, {
            queryParams: {
                tag_names: 'support_chat_deferred',
                limit: 100,
            },
        })
            .then(response => {
                setFeedIsLoading(false);
                setChats({ type: _CHATS.DEFFER, ...{ response } }, null);
            })
            .catch((error) => {
                setFeedIsLoading(false);
                setChats({ type: _CHATS.DEFFER, response: null }, error);
            });
    }

    selectMenuArchive() {
        const { setSelectList, getHistory } = this.props;

        location.hash = NAVIGATION.ROUTES.CHATS.uri;
        this.request.abort();
        setSelectList(_CHATS.ARCHIVE, null);
        getHistory({}, _CHATS.ARCHIVE);
    }

    openAddConfirm(openConfirm: boolean, el, group) {
        this.setState({
            isAddConfirmOpen: openConfirm,
            isAddAction: !this.fvLines.match(group, el?.name),
            group,
        });

        if (!openConfirm) {
            this.chosenItem = null;
        } else {
            this.chosenItem = el;
        }
    }

    checkPermission() {
        Notification.requestPermission((permission) => {
            if (permission === NOTIFY_PERMISSION.denied) {
                alert('Разрешите получать уведомления о новых чатиках');
            }
        });
    }

    onAcceptConfirm() {
        const { group, isAddAction } = this.state;

        this.setState({ isAddConfirmOpen: false });
        const itemName = this.chosenItem && `${group}__${this.chosenItem.name}` || '';

        const _item = {
            name: this.chosenItem?.name,
            display_name: this.chosenItem?.display_name,
        };

        const logData = { //log data for only chats
            event_type: EVENT_TYPES.CLICK_BUTTON,
            button_name: FAVOURITES,
            meta: {
                fav_action: FAV_ACTION.ADD,
                type: FAV_LINES_TYPES.chats,
                group,
                _item,
            },
        };

        if (isAddAction) {
            const item: IFavLineItem = this.chosenItem && group ? {
                display_name: this.chosenItem.display_name,
                chats: {
                    urgent: this.chosenItem?.chats?.urgent?.length || 0,
                },
                group,
            } : {} as IFavLineItem;

            this.chosenItem?.name && this.fvLines.add(itemName, item);
            logData.meta.fav_action = FAV_ACTION.ADD;
            this.checkPermission();
        } else {
            this.chosenItem?.name && this.fvLines.remove(itemName);
            logData.meta.fav_action = FAV_ACTION.REMOVE;
        }

        this.log?.send({ data: logData });

    }

    render() {
        const { selectedMenu, selectedMenuItem, lineIsOpen, linesListData, toggle } = this.props;
        const { group, isAddConfirmOpen } = this.state;
        const confirmQuestion = this.fvLines.getConfirmQuestion(group, this.chosenItem?.name);
        const selectedItems = {
            menuItem: selectedMenu,
            subMenuItem: selectedMenuItem,
        };

        const additionalMenus: IAdditionalMenu[] = [
            {
                type: _CHATS.DEFFER,
                title: DEFER_TITLE,
                onClick: this.selectMenuDeffer.bind(this),
            },
            {
                type: _CHATS.ARCHIVE,
                title: ARCHIVE_TITLE,
                onClick: this.selectMenuArchive.bind(this),
            },
        ];

        return <div className={`${chatStyle.chats_column} ${style.component}`}>
            <ChatToggle isOpen={lineIsOpen} toggle={toggle.bind(this)}/>

            {lineIsOpen ? <>
                <LinesList linesListData={linesListData || {} as LinesMenuType}
                           selectedItems={selectedItems}
                           onSelectLine={this.selectMenu.bind(this)}
                           openFavouritesConfirm={this.openAddConfirm.bind(this, true)}
                           additionalMenus={additionalMenus}/>
            </>
                : null
            }
            {
                isAddConfirmOpen
                    ? <Confirm onClose={this.openAddConfirm.bind(this, false)}
                               question={confirmQuestion}
                               error={null}
                               accept={this.onAcceptConfirm.bind(this)}/>
                    : null
            }
        </div>;
    }
}
