import React from 'react';
import { renderToString } from 'react-dom/server';
import ReactTooltip from 'react-tooltip';

import { EMPTY_DATA } from '../../../constants';
import { CarInfoHandler } from '../../../models/car';
import SessionsHistoryItem, { SessionHistoryInfoHandler } from '../../../models/session';
import FormatDate from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import { IBlockRules } from '../../../utils/IBlockRules';
import { isValidSessionId } from '../../../utils/isValidSessionId';
import CarNumber from '../../CarNumber';
import { Copy } from '../../Copy';
import { SessionOffer } from '../../SessionCard/SessionOffer';
import { copySessionText } from '../utils';
import * as style from './index.css';

interface IShortSessionInfoBlockProps extends IBlockRules {
    sessionInfo: SessionsHistoryItem | null;
    index?: number;
    className?: string;
}

const SIB_TOOLTIP_ID = 'SIB_TOOLTIP';

export const ShortSessionInfoBlock = React.memo((props: IShortSessionInfoBlockProps) => {
    const { sessionInfo, index = 0, className } = props;
    const sih = SessionHistoryInfoHandler;
    const car_id = sih.getCarId.call(sessionInfo, index);
    const session_id = sih.getSessionId.call(sessionInfo, index);
    const carNumber = sih.getCarNumber.call(sessionInfo, index);
    const user = sih.getUserDetails.call(sessionInfo, index);
    const carInfo = sih.getCar.call(sessionInfo, index);
    const osago_link = CarInfoHandler.getOsagoLink.call(carInfo);
    const hrMileage = sih.getHrMileage.call(sessionInfo, index);

    return car_id ? <div className={`${className} ${style.container}`}>
        <div className={style.head}>
            <div className={style.car_number}>
                <CarNumber carInfo={sih.getCar.call(sessionInfo, index)}/>
            </div>
            <div>
                <Link href={`#/cars/${car_id}/tags`}
                      target={'_blank'}>Теги</Link>
            </div>
            <div>
                {!isValidSessionId(session_id)
                    ? <div className={style.warning}>
                        Потенциально опасный session id:
                        <Copy>{session_id}</Copy>
                    </div>
                    : <Link href={`#/session/${session_id}/billing`}
                            target={'_blank'}>Сессия</Link>
                }
            </div>
        </div>
        <Copy className={style.copy} text={copySessionText(user, carNumber)}>
            <div><strong>Модель: </strong> <span className={style.car_item_value}>
                {sih.getModelName.call(sessionInfo, index)}
            </span></div>
            <div><strong>Регион: </strong> <span className={style.car_item_value}>
                {sih.getCity.call(sessionInfo, index)}
            </span></div>
            <div><strong>Статус: </strong> <span className={style.car_item_value}>
                {sih.getLastStatus.call(sessionInfo, index)}
            </span></div>
            <div><strong>Начало: </strong> <span className={style.car_item_value}>
                <FormatDate value={sih.getStart.call(sessionInfo, index)} withSecond/>
            </span></div>
            <div><strong>Конец: </strong> <span className={style.car_item_value}>
                <FormatDate value={sih.getFinish.call(sessionInfo, index)} withSecond/>
            </span></div>
            <div className={style.offer_field}
                 data-tip={OfferDetailsTooltip(sessionInfo, props.BlockRules)}
                 data-for={SIB_TOOLTIP_ID + session_id}>
                <strong>Оффер:</strong>
                <span className={`${style.car_item_value} ${style.offer_name}`}>
                    {sih.getOfferName.call(sessionInfo, index)}
                </span>
            </div>
            <div><strong>Расстояние: </strong> <span className={style.car_item_value}>{hrMileage}</span></div>
            <div><strong>Способ оплаты: </strong> <span className={style.car_item_value}>
                {sih.getSelectedCharge.call(sessionInfo, index)}
            </span></div>
            <div><strong>Итого: </strong> <span className={style.car_item_value}>
                {sih.getTotalPrice.call(sessionInfo, index)}
            </span>
            </div>
            {carNumber
            && <Link href={`https://search.yandex-team.ru/stsearch?text=${
                encodeURIComponent(carNumber)
            }`}
                     target={'_blank'}>Трекер: поиск</Link>}
            {props.BlockRules?.DownloadInsurance
                ? <Link href={osago_link} target={'_blank'} className={style.osago}>ОСАГО</Link>
                : null
            }
        </Copy>

        <ReactTooltip id={SIB_TOOLTIP_ID + session_id} html={true} />
    </div> : <>{EMPTY_DATA}</>;
});

const OfferDetailsTooltip = (sessionInfo, BlockRules) => {
    const SIH = SessionHistoryInfoHandler;
    const current_offer = SIH.getCurrentOffer.call(sessionInfo);
    const offer_proto = SIH.getOffer.call(sessionInfo);
    const total = SIH.getTotalPrice.call(sessionInfo);

    return renderToString(
        <SessionOffer BlockRules={BlockRules}
                      current_offer={current_offer}
                      offer_proto={offer_proto}
                      total={total}/>,
    );
};
