import React from 'react';

import { LSSettingItems } from '../../../types';
import LS from '../../utils/localStorage/localStorage';
import { Request2 } from '../../utils/request';
import { cgiParser } from '../ChatsCommon/chatUtils';
import * as chatsCommonStyle from '../ChatsCommon/index.css';
import { _CHATS } from '../ChatsCommon/types';
import { ErrorSource, logError } from '../Content/initErrorCounter';
import { ChatError } from './ChatError/component';
import ChatInfo from './ChatInfo';
import ChatWindow from './ChatWindow';
import { ARCHIVE_USER_SUPPORT_CHAT_TAG } from './constants';
import Feed from './Feed';
import LinesList from './LinesList';
import { CHAT_REQUESTS, REQUESTS } from './request';

interface IChatsState {
    [key: string]: any;

    lineIsOpen: boolean;
    feedIsOpen: boolean;
    error: null | Error;
}

export class Chats extends React.Component<any, IChatsState> {
    ls = new LS();
    state: IChatsState = {
        lineIsOpen: true,
        feedIsOpen: true,
        error: null,
    };
    request = new Request2({ requestConfigs: CHAT_REQUESTS });

    componentDidMount(): void {
        this.setState({
            lineIsOpen: this.ls.get(LSSettingItems.chatSettings).lineIsOpen,
            feedIsOpen: this.ls.get(LSSettingItems.chatSettings).feedIsOpen,
        });

        this.matchParamsForChats();
    }

    componentDidUpdate(prevProps: Readonly<any>, prevState: Readonly<IChatsState>, snapshot?: any) {
        const { location } = this.props;
        const newSearch = location && location.search;
        const oldSearch = prevProps.location && prevProps.location.search;
        if (newSearch !== oldSearch) {
            this.matchParamsForChats();
        }
    }

    componentDidCatch(error: Error, errorInfo): void {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);
        this.setState({
            error,
        });
    }

    matchParamsForChats() {
        const { location } = this.props;
        const [user_id, chat_id, performer_id] = cgiParser(location);
        if (user_id && chat_id) {
            this.getList([{ user_id, chat_id }], _CHATS.CUSTOM);
        } else if (performer_id) {
            this.getHistory({ performer_id }, _CHATS.CUSTOM);
        } else if (user_id) {
            this.getUserChats({ user_id }, _CHATS.CUSTOM);
        }
    }

    getList(chats: any[], type?: any) {
        const { setFeedIsLoading, setChats } = this.props;

        if (chats?.length) {
            setFeedIsLoading(true);
            this.request.abort();
            this.request.exec(REQUESTS.LIST, {
                body: { chats },
            })
                .then(response => {
                    setFeedIsLoading(false);
                    this.formatData(response, type);
                })
                .catch((error) => this.saveError(error));
        } else {
            setChats({ chats: [] });
        }
    }

    getHistory(user, type) {
        const { setFeedIsLoading } = this.props;
        setFeedIsLoading(true);
        this.request.exec(REQUESTS.ARCHIVE, {
            queryParams: Object.assign({}, user, { tags: ARCHIVE_USER_SUPPORT_CHAT_TAG }),
        })
            .then(response => {
                const chats = response?.tags
                    ?.filter(tag => {
                        return tag?.tag_details?.topic_link;
                    }).map(tag => {
                        return { user_id: tag?.object_id, chat_id: tag.tag_details.topic_link };
                    });
                setFeedIsLoading(false);
                this.getList(chats, type);
            })
            .catch((error) => this.saveError(error));
    }

    getUserChats(user, type) {
        const { setFeedIsLoading } = this.props;
        setFeedIsLoading(true);
        this.request.exec(REQUESTS.GET_LIST, {
            queryParams: user,
        })
            .then(response => {
                setFeedIsLoading(false);
                this.formatData(response, type);
            })
            .catch((error) => this.saveError(error));
    }

    async formatData(response, type) {
        const { setSelectList, setChats, selectChatItem } = this.props;
        const _response = {
            chats: {
                urgent: response?.chats,
            },
            tags: response?.chats?.map(tag => tag?.tag_data || {}),
            users: response.users,
        };

        const chat = response.chats[0];

        setSelectList(type || _CHATS.CUSTOM, null);
        setChats({ type: type || _CHATS.CUSTOM, ...{ response: _response } }, null);

        response?.chats?.length && selectChatItem(chat);
    }

    saveError(error) {
        this.props.setFeedIsLoading(false);
        this.setState({ error });
    }

    toggle(field) {
        this.setState((prev) => ({
            ...prev,
            [field]: !prev[field],
        }), () => {
            this.ls.set(LSSettingItems.chatSettings, this.state);
        });
    }

    clearError() {
        this.setState({
            error: null,
        });
    }

    render() {
        const { lineIsOpen, feedIsOpen, error } = this.state;
        const compileStyle = `__chats ${chatsCommonStyle.chats}`
            + ` ${lineIsOpen ? `${chatsCommonStyle.line_is_open}` : ''}`
            + ` ${feedIsOpen ? `${chatsCommonStyle.feed_is_open}` : ''}`
            + ` ${error ? `${chatsCommonStyle.error}` : ''}`;

        return <div className={compileStyle}>
            {error
                ? <ChatError error={error} clearError={this.clearError.bind(this)}/>
                : null
            }
            <LinesList toggle={this.toggle.bind(this, 'lineIsOpen')}
                       lineIsOpen={lineIsOpen}
                       getHistory={this.getHistory.bind(this)}/>
            <Feed toggle={this.toggle?.bind(this, 'feedIsOpen')} feedIsOpen={feedIsOpen}/>
            <ChatWindow/>
            <ChatInfo/>
        </div>;
    }
}
