import moment from 'moment';
import React from 'react';

import { ITagsWithPropsExport, UserTags } from '../../../decorators/UserTags';
import { EMPTY_DATA, ONE_SECOND, TIMERS } from '../../constants';
import { SessionHistoryInfoHandler } from '../../models/session';
import UserInfo, { UserInfoHandler } from '../../models/user';
import { Button } from '../../ui/Button';
import { ShortError } from '../../ui/ErrorBar';
import FormatDate from '../../ui/FormatDate';
import { Link } from "../../ui/Link";
import { isObjectEqual } from '../../utils/isObjectEqual';
import { NAVIGATION } from '../../utils/navigation';
import { Request2 } from '../../utils/request';
import ActiveChatsCounter from '../ActiveChatsCounter/';
import { UserCardLinks } from '../Chats/ChatInfo/UserCardLinks';
import ShortSessionInfoBlock from '../Chats/ShortSessionInfoBlock';
import { ShortUserInfoBlock } from '../Clients/ShortUserInfoBlock/component';
import UserSidebarTags from '../Clients/UserCardSidebar/UserSidebarTags';
import UserFeaturesBlock from '../Clients/UserFeaturesBlock';
import UserIdentificationBlock from '../Clients/UserIdentificationBlock';
import { Copy } from '../Copy';
import Spin from '../Spin';
import { chatLink } from './helpers';
import * as style from './index.css';
import { requestConfigs, REQUESTS } from './request';

const trackerIcon = require(`./../../../images/tracker.png`);

const LAST_5_SESSIONS = 5;
const USER_INFO_INDEX = 0;
const SESSIONS_INDEX = 1;
const TAGS_INDEX = 2;
const FULFILLED_STATUS = 'fulfilled';

interface IChatInfoProps extends ITagsWithPropsExport {
    userId: string;
    selectedChatItem: any;
    hideLastSession?: boolean;
}

interface IChatInfoState {
    isLoading: boolean;
    error: Error | null;
    userInfo: UserInfo;
    sessionsError: Error | null;
    sessions: any;
    tagsError: Error | null;
    tags: any;
}

@UserTags()
export class ChatInfoCommon extends React.Component<IChatInfoProps, IChatInfoState> {
    state: IChatInfoState = {
        isLoading: false,
        error: null,
        userInfo: {},
        sessionsError: null,
        sessions: {},
        tagsError: null,
        tags: [],
    };
    request = new Request2({ requestConfigs });
    timeout;

    componentDidMount(): void {
        this.getData(true);
    }

    shouldComponentUpdate(
        nextProps: Readonly<IChatInfoProps>,
        nextState: Readonly<IChatInfoState>,
        nextContext: any,
    ): boolean {
        return !isObjectEqual(this.state, nextState) || this.props.userId !== nextProps.userId;
    }

    componentDidUpdate(prevProps: Readonly<IChatInfoProps>, prevState: Readonly<IChatInfoState>, snapshot?: any): void {
        if (this.props.userId !== prevProps.userId) {
            this.setState({
                userInfo: {},
                sessions: {},
                tags: [],

                error: null,
                sessionsError: null,
                tagsError: null,
            });

            this.getData(true);
        }
    }

    componentWillUnmount(): void {
        this.request.abort();
        clearTimeout(this.timeout);
    }

    getData(needSpinner: boolean) {
        if (needSpinner) {
            this.setState({
                isLoading: true,
            }, () => {
                this.resolvePromises();
            });
        } else {
            this.resolvePromises();
        }
    }

    async resolvePromises() {
        this.request.abort();
        clearTimeout(this.timeout);
        const user_id = this.props.userId;

        const queue = user_id
            ? [
                this.request.exec(REQUESTS.GET_USER_INFO, { queryParams: { user_id } }),
                this.request.exec(REQUESTS.GET_USER_SESSIONS, { queryParams: { user_id, numdoc: LAST_5_SESSIONS } }),
                this.props?.getTagsPromise?.({ object_id: user_id }),
            ]
            : [];

        if (Promise.allSettled) {
            Promise.allSettled(queue).then((response: any) => {
                this.setResponseData(response);
            });
        } else {
            try {
                const response = await Promise.all(queue);
                this.setResponseData(response);
            } catch (error) {
                this.setState({ error });
            }
        }
    }

    setResponseData(response) {
        let userInfo = this.state.userInfo as Record<string, any>,
            sessions = this.state.sessions,
            tags = this.state.tags,
            isLoading = false,
            error = null, tagsError = null;
        const sessionsError = null;

        response.forEach((resp, ind) => {
            if (resp.status === FULFILLED_STATUS) {
                switch (ind) {
                case USER_INFO_INDEX:
                    return userInfo = resp.value || {};
                case SESSIONS_INDEX:
                    return sessions = resp.value || {};
                case TAGS_INDEX:
                    return tags = resp.value.records || [];
                default:
                    return;
                }
            } else {
                switch (ind) {
                case USER_INFO_INDEX:
                    return error = resp.reason;
                case SESSIONS_INDEX:
                    let sessionsError = null;

                    if (resp.reason.message !== 'The user aborted a request.') {
                        sessionsError = resp.reason;
                    }

                    return sessionsError;
                case TAGS_INDEX:
                    return tagsError = resp.reason;
                default:
                    return;
                }
            }
        });

        if (this.props.userId !== userInfo?.id) {
            userInfo = {};
            sessions = {};
            tags = [];
            isLoading = true;
        }

        this.setState({
            isLoading, userInfo, error, sessions, sessionsError, tags, tagsError,
        }, () => this.repeat());
    }

    repeat() {
        this.timeout = setTimeout(this.getData.bind(this), TIMERS.CHATS);
    }

    getCopyText(): string {
        const { userInfo, sessions } = this.state;
        const uih = UserInfoHandler;
        const sih = SessionHistoryInfoHandler;

        const phone = uih.getSimplePhone.call(userInfo);
        const debt = uih.getDebt.call(userInfo);
        const tagName = this.props.selectedChatItem?.tag || this.props.selectedChatItem?.tag_data?.tag;
        const tagInfo = this.state.tags?.find(el => el.tag === tagName) ?? {};
        const tagDisplayName = tagInfo.display_name;
        const isDispatcher = location.hash.includes(NAVIGATION.ROUTES?.OUTGOING?.uri);
        const link = chatLink(this.props?.selectedChatItem || {}, isDispatcher);
        const number = sih.getCarNumber.call(sessions);
        const offer = sih.getOfferName.call(sessions);
        const status = sih.getLastEventsStatus.call(sessions);
        const events = sih.getEvents.call(sessions) || [];
        const start = moment(events[0]?.timestamp * ONE_SECOND) ?? 0;
        const duration = start ? moment().from(start)?.replace('через ', '') : EMPTY_DATA;
        const userId = uih.getId.call(userInfo);

        const userData = `${'`' + userId + '`'} ${phone}, задолженность: ${debt}.`;
        const sessionData = `Тариф "${offer}", ${number}, ${status ? `статус ${status} (${duration})` : ''}`;
        const taskData = `Очередь ${tagDisplayName ?? tagName} \n${link}`;

        return `${userData} \n${sessionData} \n${taskData}`;
    }

    getTrackerLink(): string {
        const { userInfo } = this.state;
        const userId = UserInfoHandler.getId.call(userInfo);

        return `https://search.yandex-team.ru/stsearch?text=${userId}`;
    }

    render() {
        const notChats = !location.hash.includes(NAVIGATION.ROUTES?.CHATS?.uri);

        return <>
            {this.state.error
                ? <ShortError error={this.state.error}/>
                : null
            }
            {this.state.isLoading
                ? <Spin size={'s'}/>
                : <>
                    {notChats
                        ? <h4>
                            <div className={style.short_name}>{UserInfoHandler.getPrintName.call(this.state.userInfo)}
                            </div>
                        </h4>
                        : null
                    }
                    <ShortUserInfoBlock userInfo={this.state.userInfo}
                                        chatTagId={this.props?.selectedChatItem?.id}
                                        tags={this.state.tags}/>
                    <ActiveChatsCounter tags={this.state.tags}
                                        user_id={this.props.userId}
                                        isLoading={this.state.isLoading}/>
                    {notChats
                        ? <Copy text={this.getCopyText()} className={style.copy}>
                            <Button className={style.copy_button} basic={true}> Информация о чате </Button>
                        </Copy>
                        : null
                    }

                    <div className={style.update_time}>
                        Актуально на <FormatDate onlyTime={true} withSecond={true} value={new Date()}/>
                    </div>

                    <Button basic={true} className={style.tracker}>
                        <Link href={this.getTrackerLink()} target={'_blank'}>
                            <img src={trackerIcon}
                                 alt={''}/>
                        Трекер:поиск по id
                        </Link>
                    </Button>

                    <UserCardLinks userId={this.props.userId} title={`Карточка пользователя 👤`}/>
                    <UserFeaturesBlock user_id={this.props.userId}/>
                    <UserIdentificationBlock user_id={this.props.userId}/>
                    <h4>Работа с авто 🏎️</h4>
                </>
            }

            {this.state.sessionsError
                ? <ShortError error={this.state.sessionsError}/>
                : null
            }
            {!this.props.hideLastSession && this.state.sessions?.sessions?.map((item, index) => {
                const finish = SessionHistoryInfoHandler.getFinish.call(this.state.sessions, index);

                return (index === 0 || !finish)
                    && <ShortSessionInfoBlock sessionInfo={this.state.sessions}
                                              key={index}
                                              index={index}/> || null;
            })
            }

            {this.state.tagsError
                ? <ShortError error={this.state.tagsError}/>
                : null
            }
            {this.state.isLoading
                ? <Spin size={'s'}/>
                : <UserSidebarTags userInfo={this.state.userInfo} tags={this.state.tags}/>
            }
        </>;
    }
}
