import React from 'react';

import { ROBOT_FRONTEND, STAFF_ONLY } from '../../../constants';
import { Confirm } from '../../../ui/FullModal';
import { NAVIGATION } from '../../../utils/navigation';
import { Request2 } from '../../../utils/request';
import { CHAT_REQUESTS, REQUESTS } from '../../Chats/request';
import { ErrorSource, logError } from '../../Content/initErrorCounter';
import { chatLink } from '../helpers';
import { NChats } from '../types';
import { CommonMessageContainer } from './CommonMessageContainer';
import { isViewed } from './CommonMessageFooter';
import MESSAGE_TYPE = NChats.MESSAGE_TYPE;

const BANK_CARD_REPLACE_TEXT = `[i]Здесь было ваше сообщение с номером банковской карты. Для вашей безопасности мы его удалили. Для проверки платежа нам не нужен полный номер, достаточно маски карты (первые 6 и последние 4 цифры).[/i]`;

export class CommonMessage extends React.Component<any, any> {
    state = {
        confirmIsShowing: false,
        question: '',
        accept: () => {},
        error: null,
        isWorking: false,
        extraError: null,
    };

    request = new Request2({ requestConfigs: CHAT_REQUESTS });

    componentDidCatch(error: Error, errorInfo: React.ErrorInfo) {
        logError(error, ErrorSource.ERROR_BOUNDARY, errorInfo);
        this.setState({
            extraError: error,
        });
    }

    closeConfirm() {
        this.setState({
            confirmIsShowing: false,
        });
    }

    removeMessage() {
        this.setState({
            confirmIsShowing: true,
            question: 'Удалить сообщение, пока пользователь не увидел этого безобразия?',
            accept: () => {
                this.setState({
                    isWorking: true,
                }, () => {
                    this.request.exec(REQUESTS.EDIT_MESSAGE, {
                        body: {
                            user_id: this.props.selectedChatItem?.originator,
                            chat_id: this.props.selectedChatItem?.id,
                            message_id: this.props.item && this.props.item.id,
                            delete: 'da',
                        },
                    })
                        .then(() => {
                            this.setState({
                                confirmIsShowing: false,
                                isWorking: false,
                            });
                        })
                        .catch(error => {
                            this.setState({
                                error,
                                isWorking: false,
                            });
                        });
                });
            },
        });
    }

    replaceMessage() {
        this.setState({
            confirmIsShowing: true,
            question: <><h3>Внимание!</h3><div>Сообщение у клиента затрётся с заглушкой:</div><br/>
                <div><i>{BANK_CARD_REPLACE_TEXT}</i></div>
                <br/><div>Вы уверены что здесь содержится номер банковской карты в открытом
                виде или другие приватные данные карты?</div></>,
            accept: () => {
                const body = {
                    user_id: this.props.selectedChatItem?.originator,
                    chat_id: this.props.selectedChatItem?.id,
                    message_id: this.props.item && this.props.item.id,
                    text: BANK_CARD_REPLACE_TEXT,
                    type: MESSAGE_TYPE.PLAINTEXT,
                    force: true,
                };
                this.request.exec(REQUESTS.EDIT_MESSAGE, { body })
                    .then(() => {
                        this.setState({
                            isWorking: false,
                            confirmIsShowing: false,
                        });
                    })
                    .catch((error) => {
                        this.setState({
                            isWorking: false,
                            error,
                        });
                    });
            },
        });
    }

    componentDidMount(): void {
        this.request.abort();
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getRenderData() {
        const { item, chat, selectedChatItem } = this.props;
        const isOriginator = item?.author === selectedChatItem?.originator;

        return {
            isOriginator,
            author: chat?.users?.[item?.author] || item?.author || null,
            isRobot: item?.author === ROBOT_FRONTEND,
            isStaffOnly: Array.isArray(item?.traits) && item.traits.includes(STAFF_ONLY),
            isViewedMessage: isViewed({ id: item?.id, user_last_viewed: chat?.user_last_viewed }),
            href: createHref(isOriginator, item?.author),
        };
    }

    render() {
        const { isOriginator, isRobot, isStaffOnly, author, href, isViewedMessage } = this.getRenderData();
        const { children, chat, item, isSafeMode } = this.props;
        const chatLinkText = chatLink(this.props.selectedChatItem);

        return <>
            <CommonMessageContainer item={item}
                                    isRobot={isRobot}
                                    isOriginator={isOriginator}
                                    children={children}
                                    author={author}
                                    chat={chat}
                                    chatLinkText={chatLinkText}
                                    extraError={this.state.extraError}
                                    href={href}
                                    isSafeMode={isSafeMode}
                                    isStaffOnly={isStaffOnly}
                                    isViewedMessage={isViewedMessage}
                                    isSelected={this.props.isSelected}
                                    changeSelectedMessages={this.props.changeSelectedMessages.bind(this, item.id)}
                                    removeMessage={this.removeMessage.bind(this)}
                                    replaceMessage={this.replaceMessage.bind(this)}/>

            {
                this.state.confirmIsShowing
                    ? <Confirm error={this.state.error}
                               isWorking={this.state.isWorking}
                               onClose={this.closeConfirm.bind(this)}
                               accept={this.state.accept}
                               question={this.state.question}/>
                    : null
            }
        </>;
    }
}

const createHref = (isOriginator: boolean, author: string): string => {
    return `${location.origin}${location.pathname}`
        + `#${NAVIGATION.ROUTES.CHATS.uri}`
        + `?${isOriginator ? 'user_id' : 'performer_id'}=${author}`;
};
