import moment from 'moment';
import React from 'react';

import { ONE_DAY, ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import { Request2 } from '../../../utils/request';
import { FormConstructor } from '../../FormConstructor';
import { controlType, ISchemaItem, SchemaItemVisual } from '../../FormConstructor/types';
import { SimpleError } from '../../SimpleError';
import * as style from '../index.css';
import { requestConfigs, REQUESTS } from '../request';

const ONE = 1;
const TWO = 2;
const FIVE = 5;
const NINE = 9;
const TEN = 10;
const TWENTY = 20;
const THIRTY = 30;

interface IDeferModalProps {
    onClose: () => void;
    onDeferred: () => void;
    selectedChatItem: any;
    config?: any;
    deferUntilMorning?: boolean;
}

interface IDeferModalState {
    isLoading: boolean;
    error: Error | null;
    body: any;
    isValid: boolean;
}

export class DeferModal extends React.Component<IDeferModalProps, IDeferModalState> {
    state: IDeferModalState = {
        isLoading: false,
        error: null,
        body: {} as any,
        isValid: false,
    };
    request = new Request2({ requestConfigs });

    onFormChange(body, isValid) {
        this.setState({
            body,
            isValid,
        }, () => {
            this.periodIsOver();
        });
    }

    onButtonClick(defer_until) {
        this.setState({
            body: { ...this.state.body, defer_until },
        });
    }

    defer() {
        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.DEFER, {
                queryParams: {
                    tag_id: this.getTagId(),
                },
                body: this.state.body,
            })
                .then(() => {
                    this.setState({
                        isLoading: false,
                        error: null,
                    });
                    this.props.onClose();
                    this.props.onDeferred();
                })
                .catch(error => {
                    this.setState({
                        isLoading: false,
                        error,
                    });
                });
        });
    }

    componentWillUnmount(): void {
        this.request.abort();
    }

    getTagId() {
        return this.props.selectedChatItem?.tag_data?.tag_id || this.props.selectedChatItem?.tag_id;
    }

    periodIsOver() {
        const delta = this.state.body?.defer_until * ONE_SECOND - new Date().getTime();

        return delta > ONE_DAY * this.props.config?.max_day_period;
    }

    buildSchema(): { [key: string]: ISchemaItem } {
        return {
            'tag_id': {
                'display_name': 'tag_id',
                'order': 0,
                'type': controlType.ignore,
            },
            'defer_until': {
                'display_name': 'дата',
                required: true,
                'order': 1,
                visual: SchemaItemVisual.TIMESTAMP,
                'type': controlType.numeric,
            },
            ...this.props?.config?.need_comment && {
                'comment': {
                    'display_name': 'comment',
                    'order': 2,
                    'type': controlType.text,
                },
            },
        };
    }

    getMorning(): number {
        const tomorrow = moment().endOf('day').add(1, 'second');

        return new Date().getHours() > NINE
            ? Math.round(+tomorrow.add(NINE, 'hour') / ONE_SECOND)
            : Math.round(+moment().set({ hour: NINE, minute: 0, second: 0 }) / ONE_SECOND);
    }

    getTerms(): any[] {
        const minutes = 'минут';
        const hour = 'час';
        const hours = 'часа';
        const to_morning = `до утра (${NINE} часов)`;

        const morning = this.getMorning();

        return [
            [FIVE, minutes, Math.round(+moment().add(FIVE, 'minute') / ONE_SECOND)],
            [TEN, minutes, Math.round(+moment().add(TEN, 'minute') / ONE_SECOND)],
            [TWENTY, minutes, Math.round(+moment().add(TWENTY, 'minute') / ONE_SECOND)],
            [THIRTY, minutes, Math.round(+moment().add(THIRTY, 'minute') / ONE_SECOND)],
            [ONE, hour, Math.round(+moment().add(ONE, 'hour') / ONE_SECOND)],
            [TWO, hours, Math.round(+moment().add(TWO, 'hour') / ONE_SECOND)],
            ['', to_morning, morning],
        ];
    }

    render() {
        const tag_id = this.getTagId();
        const over = this.periodIsOver();
        const defer_until = this.state.body.defer_until || this.props.deferUntilMorning && this.getMorning();

        return <Window onClose={this.props.onClose.bind(this)}
                       title={'Отложить до'}
                       className={style.defer_dialog}
                       error={this.state.error}>

            <div>
                {this.getTerms().map((el, ind) => {
                    const timeIndex = 2;
                    const title = `${el[0]} ${el[1]}`;

                    return <Button key={ind}
                                   basic={true}
                                   onClick={this.onButtonClick.bind(this, el[timeIndex])}
                                   className={style.terms}>{title}</Button>;
                })}
            </div>
            <FormConstructor onChange={this.onFormChange.bind(this)}
                             className={style.defer_dialog}
                             initialData={{ tag_id, defer_until }}
                             schema={this.buildSchema()}/>
            {
                over ? <SimpleError error={new SyntaxError('плохой период, плохой')}/> : null
            }
            <Button onClick={this.defer.bind(this)} disabled={!this.state.isValid || over}>Отложить</Button>
        </Window>;
    }
}
