import React from 'react';
import { ItemStyle } from 'react-tiny-virtual-list';

import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import FormatDate from '../../../ui/FormatDate';
import { getTagId } from '../chatUtils';
import style from '../index.css';
import { NChats } from '../types';
import { IFeedListItemDispatchToProps, IFeedListItemStateToProps } from './index';
import MESSAGE_TYPE = NChats.MESSAGE_TYPE;
import IChatListItem = NChats.IChatListItem;

export interface IFeedListItemProps extends IFeedListItemStateToProps, IFeedListItemDispatchToProps {
    item: IChatListItem;
    style: ItemStyle;
    users: any;

    markOutgoing?: boolean;
    selectedChatItem?: any;
    selectChatItem?: () => void;
}

export class FeedListItem extends React.Component<IFeedListItemProps, any> {
    checkIfImportant() {
        const { item, selectedChatItem } = this.props;

        if (getTagId(item) === getTagId(selectedChatItem)) {
            return selectedChatItem?.tag_data?.chat_meta?.important;
        }

        return Boolean(item?.tag_data?.chat_meta?.important);

    }

    render() {
        const { item } = this.props;
        const originator = item.originator;
        const author = item.last_message?.author;
        const performer_id = item.tag_data?.performer;

        const originatorName = this.props.users && UserInfoHandler.getPrintName.call(this.props.users[originator]);
        const performerUserName = performer_id
            && this.props.users
            && UserInfoHandler.getUsername.call(this.props.users[performer_id]);

        const message = item.last_message?.type === MESSAGE_TYPE.PLAINTEXT ? item.last_message?.text : '';
        const lastMessageIsOriginator = item.originator === author;

        const isSelected = this.props.selectedChatItem
            && originator === this.props.selectedChatItem?.originator
            && item.id === this.props.selectedChatItem?.id;

        const isImportantChat = this.checkIfImportant();

        return <div className={`${style.feed_item_component}`
            + ` ${lastMessageIsOriginator ? style.originator_message : style.other_message}`
            + ` ${isSelected ? style.wrap_selected : ''}`
            + ` ${!performer_id ? style.without_performer : ''}`
            + ` ${isImportantChat ? style.important_chat : ''}`
            + ` ${this.props.markOutgoing ? style.outgoing_chat : ''}`}
                    onClick={this.props.selectChatItem?.bind(this, item)}
                    style={this.props.style}>
            {isSelected ? <div className={`${style.selected}`}/> : null}
            <div className={`${style.date}`}><FormatDate value={item.last_message?.timestamp * ONE_SECOND}/></div>
            <div className={`${style.left_bar}`}/>
            <div className={`${style.right_bar}`}/>
            <div className={`${style.client}`}>{originatorName}</div>
            <div className={`${style.message}`}>{message}</div>
            <PerformerAvatar performerUserName={performerUserName}/>
            <div className={`${style.operator_name}`}>{performerUserName}</div>
            {
                item.icon
                    ? <div className={`${style.chat_name}`}
                           style={{ backgroundImage: `url(${item.icon})` }}
                           title={item.name}/>
                    : null

            }
        </div>;
    }
}

const PerformerAvatar = React.memo((props: { performerUserName }) => {
    const avatarUrl = props.performerUserName && props.performerUserName !== EMPTY_DATA
        ? `https://center.yandex-team.ru/api/v1/user/${props.performerUserName}/avatar/50.jpg`
        : `https://carsharing.s3.yandex.net/admin/static/default_performer.png`;

    return <div className={`${style.avatar}`}>
        {props.performerUserName !== EMPTY_DATA
            ? <img src={avatarUrl} alt={'avatar'} className={style.avatar}/>
            : null
        }
    </div>;
});
