import React from 'react';
import { useSelector } from 'react-redux';

import { areAllChatsLoading, areMyChatsLoading, areNewChatsLoading } from '../../../reducers/chatReducer/getters';
import { Link } from '../../../ui/Link';
import { Star } from '../../../ui/Star';
import Spin from '../../Spin';
import { ADDITIONAL_CONTROL_TYPES, IAdditionalControl, IAdditionalMenu, ISelectedMenus, LinesMenuType } from '../types';
import { BaseMenu } from './BaseMenu';
import style from './style.css';

interface ILinesListProps {
    linesListData: LinesMenuType;
    selectedItems: ISelectedMenus;
    onSelectLine: () => void;
    additionalControls?: IAdditionalControl[];
    additionalMenus?: IAdditionalMenu[];
    openFavouritesConfirm: (el: string, group: string) => void;
}

export const LinesList = React.memo((props: ILinesListProps) => {
    const {
        linesListData,
        onSelectLine,
        selectedItems,
        additionalControls,
        additionalMenus,
        openFavouritesConfirm,
    } = props;

    const updateBtn = additionalControls && additionalControls.find(el => {
        return el.type === ADDITIONAL_CONTROL_TYPES.UPDATE_BUTTON;
    });

    const loadingChats = {
        all_chats_loading: useSelector(areAllChatsLoading),
        my_chats_loading: useSelector(areMyChatsLoading),
        new_chats_loading: useSelector(areNewChatsLoading),
    };

    return <div className={`${style.component}`}>
        {updateBtn
            ? <div className={style.update}>
                <Link onClick={updateBtn.onClick.bind(null)} className={style.update_button}>
                    Обновить
                </Link>

                {updateBtn.state?.isLoading
                    ? <Spin size={'s'}/>
                    : null
                }
            </div>
            : null
        }

        {Object.entries(linesListData)?.map((lines, index) => {
            return <BaseMenu data={lines}
                             key={index}
                             isLoading={loadingChats[`${lines?.[0]}_loading`] ?? false}
                             selectedItems={selectedItems}
                             onFavClick={openFavouritesConfirm.bind(null)}
                             onMenuClick={onSelectLine.bind(null)}/>;
        })}

        {additionalMenus
            ? additionalMenus.map((menuData, index) => {
                const isMenuSelected = selectedItems.menuItem === menuData.type;

                return <React.Fragment key={index}>
                    {menuData.showSubmenu
                        ? <BaseMenu data={menuData.submenuData}
                                    selectedItems={selectedItems}
                                    onMenuClick={menuData.onClick.bind(null)}
                                    title={menuData.title}/>

                        : <div onClick={menuData.onClick.bind(null)}
                               id={style[menuData.type]}
                               key={menuData.type}
                               className={`${style.interactive} ${isMenuSelected ? style.selected : ''} ${style.menu_title}`}>
                            <Star size={'l'} className={`${style[menuData.type]}`}/>
                            {menuData.title}
                        </div>
                    }
                </React.Fragment>;
            })
            : null
        }
    </div>;
});
