import produce from 'immer';
import * as React from 'react';

import { GetSessions, IGetSessionsPropsExport } from '../../../../decorators/GetSessions';
import RetryIcon from '../../../../svg-components/retry.component.svg';
import { LSSettingItems } from '../../../../types';
import { EMPTY_DATA } from '../../../constants';
import { SessionHistoryInfoHandler } from '../../../models/session';
import { Button } from '../../../ui/Button';
import { FullModal, Window } from '../../../ui/FullModal';
import Select from '../../../ui/Select';
import LS from '../../../utils/localStorage/localStorage';
import ShortSessionInfoBlock from '../../Chats/ShortSessionInfoBlock';
import Spin from '../../Spin';
import * as style from '../index.css';
import { NChats } from '../types';
import IChatListItem = NChats.IChatListItem;
import ITopicMessageItem = NChats.ITopicMessageItem;

export const getImgPrefix = () => {
    const ls = new LS();

    return ls.envIsTesting()
        ? `https://s3.mdst.yandex.net/carsharing-chat`
        : `https://carsharing-chat.s3.yandex.net`;
};

const getMedia = (props: { resource_id: string; preview?: boolean; chat_id: string; user_id: string }) => {
    const { resource_id, chat_id, user_id, preview } = props;
    const ls = new LS();

    return fetch(`saasnew/api/support_staff/chat/resource${preview ? '/preview' : ''}`
        + `?chat_id=${chat_id}&resource_id=${resource_id}`
        + `&user_id=${user_id}`
    , {
        method: 'GET', headers: {
            saasnew: `https://${ls.get(LSSettingItems.env)}.carsharing.yandex.net`,
            admbuild: '1',
            credentials: 'include',
        },
    })
        .then(response => {
            if (!response.ok) {
                return response.text().then((error) => {
                    throw error;
                });
            }

            return response.blob();
        })
        .then(resource => {
            return new Promise(() => {
                const fileReader: any = new FileReader();

                fileReader.readAsArrayBuffer(resource);
            });
        });

};

interface IImg {
    src: string;
    selectedChat: IChatListItem;
    item: ITopicMessageItem;
    scrollToBottom?: () => void;
}

export class Img extends React.Component<IImg> {
    state: any = {
        imgIsLoading: true,
        result: {},
        error: null,

        showOriginal: false,
        originalIsLoading: true,
        originalResult: {},
    };

    componentDidMount(): void {
        this.loadImg();
    }

    loadImg() {
        this.setState(produce(this.state, draft => {
            draft.imgIsLoading = true;
        }), () => {
            const { src, selectedChat /*item*/ } = this.props;
            const _props = {
                preview: true, resource_id: src, chat_id: selectedChat.id, user_id: selectedChat.originator,
            };

            getMedia(_props)
                .then(result => {
                    this.setState(produce(this.state, draft => {
                        draft.imgIsLoading = false;
                        draft.result = result;
                        draft.error = null;
                    }), () => {
                        this.props.scrollToBottom && this.props.scrollToBottom();
                    });
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.imgIsLoading = false;
                        draft.error = error;
                    }));
                });
        });
    }

    showOriginal() {
        this.setState(produce(this.state, draft => {
            draft.showOriginal = true;
            draft.originalIsLoading = true;
        }), () => {
            const { src, selectedChat /*item*/ } = this.props;
            const _props = {
                preview: false, resource_id: src, chat_id: selectedChat.id, user_id: selectedChat.originator,
            };

            getMedia(_props)
                .then(result => {
                    this.setState(produce(this.state, draft => {
                        draft.originalIsLoading = false;
                        draft.originalResult = result;
                    }));
                });
        });
    }

    hideOriginal() {
        this.setState(produce(this.state, draft => {
            draft.showOriginal = false;
            draft.originalIsLoading = false;
        }));
    }

    reload(e: KeyboardEvent) {
        e.stopPropagation();
        this.loadImg();
    }

    render() {
        return <div className={style.img_preview_container}>
            {this.state.imgIsLoading
                ? <Spin size={'l'}/>
                : this.state.error ? <div>
                    <span title={this.state.error}
                          className={style.error_load_resource}
                          onClick={this.showOriginal.bind(this)}>error</span>
                </div>
                    : <img className={`${style.img_preview}`}
                           onClick={this.showOriginal.bind(this)}
                           src={URL.createObjectURL(this.state.result.resource)}/>}
            <RetryIcon className={style.retry}
                       onClick={this.reload.bind(this)}/>
            {
                this.state.showOriginal
                && <OriginalImage style={{ color: 'red' }}
                                  src={this.state.originalResult
                                  && this.state.originalResult.resource
                                  && URL.createObjectURL(this.state.originalResult.resource)}
                                  isLoading={this.state.originalIsLoading}
                                  onClose={this.hideOriginal.bind(this)}
                                  exifTags={this.state.originalResult.tags}/>
            }
        </div>;
    }
}

export class Video extends React.Component<IImg> {
    state: any = {
        videoIsLoading: true,
        result: {},
        error: null,

        showOriginal: false,
        originalIsLoading: true,
        originalResult: {},
    };

    componentDidMount(): void {
        this.loadVideo();
    }

    loadVideo() {
        this.setState(produce(this.state, draft => {
            draft.videoIsLoading = true;
        }), () => {
            const { src, selectedChat } = this.props;
            const _props = {
                preview: true, resource_id: src, chat_id: selectedChat.id, user_id: selectedChat.originator,
            };

            getMedia(_props)
                .then(result => {
                    this.setState(produce(this.state, draft => {
                        draft.videoIsLoading = false;
                        draft.result = result;
                        draft.error = null;
                    }));
                })
                .catch(error => {
                    this.setState(produce(this.state, draft => {
                        draft.videoIsLoading = false;
                        draft.error = error;
                    }));
                });
        });
    }

    showOriginal() {
        this.setState(produce(this.state, draft => {
            draft.showOriginal = true;
            draft.originalIsLoading = true;
        }), () => {
            const { src, selectedChat } = this.props;
            const _props = {
                preview: false, resource_id: src, chat_id: selectedChat.id, user_id: selectedChat.originator,
            };

            getMedia(_props)
                .then(result => {
                    this.setState(produce(this.state, draft => {
                        draft.originalIsLoading = false;
                        draft.originalResult = result;
                    }));
                });
        });
    }

    hideOriginal() {
        this.setState(produce(this.state, draft => {
            draft.showOriginal = false;
            draft.originalIsLoading = false;
        }));
    }

    reload(e: KeyboardEvent) {
        e.stopPropagation();
        this.loadVideo();
    }

    render() {
        const src = this.state.result && this.state.result.resource && URL.createObjectURL(this.state.result.resource);

        return <div className={style.img_preview_container}>
            {this.state.videoIsLoading
                ? <Spin size={'l'}/>
                : this.state.error ? <div>
                    <span title={this.state.error} className={style.error_load_resource}>error</span>
                </div>
                    : <video src={src}>
                        <source src={src}/>
                    </video>}
            <RetryIcon className={style.retry}
                       onClick={this.reload.bind(this)}/>
            {
                this.state.showOriginal
                && <div/>
            }
        </div>;
    }
}

const ExifData = React.memo((props: { exifTags: {}; preset?: string[] }) => {
    const data: { type: string; value: string }[] = [];
    if (props.preset) {
        props.preset.forEach(_preset => {
            switch (_preset) {
            case 'size':
                return data.push({
                    type: 'size',
                    value: `${props.exifTags['Image Height'] && props.exifTags['Image Height'].value}x${props.exifTags['Image Width'] && props.exifTags['Image Width'].value}`,
                });
            case 'date':
                return data.push({
                    type: 'date',
                    value: `${props.exifTags['DateTimeOriginal'] && props.exifTags['DateTimeOriginal'].value}`,
                });
            default:
                return null;
            }
        });
    }

    return <div className={style.exif_data_container}>
        {
            data.map((item, index) =>
                <div key={index}>
                    <strong>{item.type}</strong>: {item.value.includes('undefined') ? EMPTY_DATA : item.value}
                </div>)
        }
    </div>;
});

const OriginalImage = React.memo((props: {
    src: string;
    isLoading: boolean;
    onClose: () => void;
    style: any;
    exifTags: any;
}) => {
    return <FullModal onClose={props.onClose} isMinimal={false}>
        {props.isLoading ? <Spin/> : <div>
            <img src={props.src} className={style.original_img} style={props.style}/>
            <ExifData exifTags={props.exifTags} preset={['size', 'date']}/>
        </div>}
    </FullModal>;
});

export interface IDialogProps {
    onClose: () => void;
    selectedChat: any;
    sendMessage?: (text: string) => void;
}

export interface ISessionDialogProps extends IDialogProps, IGetSessionsPropsExport {
}

@GetSessions({ numdoc: 50 })
export class SessionDialog extends React.Component<ISessionDialogProps> {
    state = {
        selectedItem: 0,
    };

    componentDidMount(): void {
        this.props.getSessions
        && this.props.getSessions({ user_id: this.props.selectedChat.originator });
    }

    onSelect(selectedItem) {
        this.setState(produce(this.state, draft => {
            draft.selectedItem = selectedItem ?? '';
        }));
    }

    send() {
        const id = SessionHistoryInfoHandler.getSessionId.call(this.props.sessionsData, this.state.selectedItem);
        this.props.sendMessage && this.props.sendMessage(id);
        this.props.onClose();
    }

    render() {
        const { onClose, sessionError, sessionIsLoading, sessionOptions, sessionsData } = this.props;
        const { selectedItem } = this.state;

        return <Window onClose={onClose.bind(this)}
                       title={'Отправка сессии пользователю'}
                       className={style.send_session_dialog}
                       error={sessionError}>
            {
                <div>
                    <Select onSelect={this.onSelect.bind(this)}
                            disabled={sessionIsLoading}
                            placeholder={'сессия пользователя'}
                            initialValues={[selectedItem]}
                            multiSelect={false}
                            options={sessionOptions ?? []}/>
                </div>
            }
            {
                selectedItem !== undefined
                && <ShortSessionInfoBlock sessionInfo={sessionsData ?? {}}
                                          className={style.session_info}
                                          index={selectedItem}/>
            }
            {
                selectedItem !== undefined
                && <div className={style.right_button}>
                    <Button onClick={this.send.bind(this)}>Отправить</Button>
                </div>
            }
        </Window>;
    }
}
