import React, { useEffect, useState } from 'react';

import { useRequestHandler } from '../../hooks/useRequestHandler';
import { Button } from '../../ui/Button';
import { Window } from '../../ui/FullModal';
import { JsonModal } from '../../ui/FullModal/JsonModal';
import { Link } from '../../ui/Link';
import { Request2 } from '../../utils/request';
import { FormConstructor } from '../FormConstructor';
import { SimpleError } from '../SimpleError';
import Spin from '../Spin';
import style from './index.css';
import { requestConfigs, REQUESTS } from './requests';

interface ICheckFuelStatusProps {
    userId: string | undefined;
}

export const CheckFuelStatus = React.memo((props: ICheckFuelStatusProps) => {
    const [isStatusModalOpen, openStatusModal] = useState<boolean>(false);

    return <div>
        <Button onClick={openStatusModal.bind(null, true)}>Статусы АЗС</Button>

        {isStatusModalOpen
            ? <CheckFuelStatusModal userId={props.userId} openStatusModal={openStatusModal.bind(null)}/>
            : null
        }
    </div>;
});

interface ICheckFuelStatusModalProps {
    userId: string | undefined;
    openStatusModal: (isOpened: boolean) => void;
}

interface IColumnData {
    id: string;
    status: string;
    fuel?: string;
    lites?: number;
    error?: Error;
}

const CheckFuelStatusModal = React.memo((props: ICheckFuelStatusModalProps) => {
    const { openStatusModal, userId } = props;

    const request = new Request2({ requestConfigs });

    const options = React.useMemo(() => {
        return {
            requestName: REQUESTS.GET_COLUMNS_STATUS,
            requestOptions: {
                queryParams: { scheme: true },
            },
        };
    }, []);

    const [isLoading, response, error, getFuelStatus] = useRequestHandler<any>(request, options);

    React.useEffect(() => {
        getFuelStatus();
    }, []);

    const [columns, setColumns] = useState<IColumnData[] | undefined>();
    const [columnData, setColumnData] = useState<IColumnData | undefined>();
    const [colError, setColError] = useState<Error | null>(null);
    const [colLoading, setColLoading] = useState<boolean>(false);
    const [formData, changeFormData] = useState<any>();

    useEffect(() => {
        formData?.station_id ? getStationInfo() : null;
    }, [formData?.station_id]);

    function getStationInfo() {
        setColLoading(true);
        request.exec(REQUESTS.GET_COLUMNS_STATUS, {
            queryParams: { station_id: formData?.station_id },
        })
            .then(response => {
                setColumns(response.columns);
                setColLoading(false);
                setColError(null);
            })
            .catch(colError => {
                setColumns([]);
                setColLoading(false);
                setColError(colError);
            });
    }

    return <Window onClose={openStatusModal.bind(null, false)}
                   title={'проверка статусов АЗС'}
                   error={error}>

        <div className={style.data}>
            {isLoading
                ? <Spin/>
                : response
                    ? <FormConstructor schema={response?.cgi?.structure || {}}
                                       onChange={changeFormData.bind(null)}
                                       initialData={{ user_id: userId }}
                                       className={style.fc}
                                       hideChanges={true}/>
                    : <div>Не удалось получить форму</div>
            }

            {colLoading
                ? <Spin/>
                : colError
                    ? <SimpleError error={colError}/>
                    : columns?.length
                        ? <table>
                            <thead className={style.header}>
                                <tr>
                                    <td>Id</td>
                                    <td>Status</td>
                                    <td>Fuel</td>
                                    <td>Lites</td>
                                    <td>Details</td>
                                </tr>
                            </thead>

                            <tbody>
                                {columns.map((column, ind) => {
                                    return <tr className={style.column_item} key={column.id + ind}>
                                        <td>{column.id}</td>
                                        <td>{column.status}</td>
                                        <td>{column.fuel ?? ''}</td>
                                        <td>{column.lites ?? ''}</td>
                                        <td>
                                            <Link onClick={setColumnData.bind(null, column)}>
                                            json
                                            </Link>
                                        </td>
                                    </tr>;
                                })
                                }
                            </tbody>
                        </table>
                        : columns && <div>Ничего не нашлось:(</div>
            }

            {columnData
                ? <JsonModal obj={columnData} onClose={setColumnData.bind(null, null)}/>
                : null
            }

        </div>
    </Window>;
});
