import * as React from 'react';

import { Collapse2 } from '../../../ui/Collapse2';
import { Wallet } from '../../../ui/Wallet';
import { CSSettingsItems, CustomSettings } from '../../../utils/customSettings';
import { Request2 } from '../../../utils/request';
import { AccountDetailsModal } from '../../Settings/Wallets/AccountDetailsModal';
import { IAccount } from '../../Settings/Wallets/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import { BillingViewActivateModal } from './BillingViewActivateModal';
import { BillingViewAddModal } from './BillingViewAddModal';
import { BillingViewDeleteModal } from './BillingViewDeleteModal';
import * as styles from './index.css';

interface IBillingViewAccountsState {
    [key: number]: string;

    userAccounts: IAccount[];
    error: Error | null;
    walletsLoading: boolean;
    showAddModal: boolean;
    showEditModal: boolean;
    showActivateModal: boolean;
    showDeleteModal: boolean;
    accountToEdit?: IAccount;
}

interface IBillingViewAccountsProps {
    userId: string;
}

export class BillingViewAccounts extends React.Component<IBillingViewAccountsProps, IBillingViewAccountsState> {
    cs = new CustomSettings();

    state: IBillingViewAccountsState = {
        userAccounts: [],
        error: null,
        walletsLoading: true,
        showAddModal: false,
        showEditModal: false,
        showActivateModal: false,
        showDeleteModal: false,
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    fetchWallets() {
        const user_id = this.props.userId;

        this.request.exec(REQUESTS.GET_ACCOUNTS, {
            queryParams: {
                user_id,
            },
        })
            .then((response) => {
                const { accounts } = response;

                this.setState({
                    userAccounts: accounts,
                    walletsLoading: false,
                });
            })
            .catch((walletsError) => {
                this.setState({
                    error: walletsError,
                    walletsLoading: false,
                });
            });
    }

    changeState(key, value) {
        this.setState({
            [key]: value,
        });
    }

    componentDidMount() {
        this.fetchWallets();
    }

    componentDidUpdate(prevProps: Readonly<IBillingViewAccountsProps>, prevState: Readonly<IBillingViewAccountsState>) {
        if (this.props.userId !== prevProps.userId) {
            this.fetchWallets();
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const {
            walletsLoading,
            showAddModal,
            showEditModal,
            accountToEdit,
            userAccounts,
            showDeleteModal,
            showActivateModal,
            error,
        } = this.state;

        const { userId } = this.props;

        return (
            <>
                {showAddModal && (
                    <BillingViewAddModal reloadData={this.fetchWallets.bind(this)}
                                         user_id={userId}
                                         onClose={this.changeState.bind(this, 'showAddModal', false)}/>
                )}
                {showEditModal && accountToEdit && (
                    <AccountDetailsModal reloadData={this.fetchWallets.bind(this)}
                                         account={accountToEdit}
                                         onClose={this.changeState.bind(this, 'showEditModal', false)}/>
                )}
                {showDeleteModal && accountToEdit && (
                    <BillingViewDeleteModal user_id={userId}
                                            account={accountToEdit}
                                            onClose={this.changeState.bind(this, 'showDeleteModal', false)}
                                            reloadData={this.fetchWallets.bind(this)}/>
                )}
                {showActivateModal && accountToEdit && (
                    <BillingViewActivateModal account={accountToEdit}
                                              onClose={this.changeState.bind(this, 'showActivateModal', false)}
                                              reloadData={this.fetchWallets.bind(this)}/>
                )}
                <Collapse2 initialExpanded={this.cs.get(CSSettingsItems.walletsExpanded)}
                           title={'Кошельки'}
                           expandText={'Развернуть кошельки'}
                           className={`${styles.cardContainer} ${error || walletsLoading ? styles.displayBlock : ''}`}
                           headerClassname={styles.accountsHeader}>
                    {walletsLoading
                        ? <Spin/>
                        : <>
                            <div className={styles.addButton}
                                 onClick={this.changeState.bind(this, 'showAddModal', true)}>
                                <div className={styles.addButtonText}>
                                +
                                </div>
                            </div>
                            {error
                                ? <SimpleError error={error}/>
                                : userAccounts.map((account, key) => {
                                    return <Wallet key={key}
                                                   active={account.is_actual}
                                                   account={account}
                                                   onEdit={this.setState.bind(this, {
                                                       showEditModal: true,
                                                       accountToEdit: account,
                                                   })}
                                                   onDelete={this.setState.bind(this, {
                                                       showDeleteModal: true,
                                                       accountToEdit: account,
                                                   })}
                                                   setActive={this.setState.bind(this, {
                                                       showActivateModal: true,
                                                       accountToEdit: account,
                                                   })}/>;
                                })
                            }
                        </>}
                </Collapse2>
            </>
        );
    }
}
