import * as React from 'react';

import { Button } from '../../../ui/Button';
import Checkbox from '../../../ui/Checkbox';
import { Window } from '../../../ui/FullModal';
import Select, { IOptionInfo } from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import * as styles from './BillingViewAddModal.css';

interface IBillingViewAddModalProps {
    onClose: () => void;
    reloadData: () => void;
    user_id: string;
}

interface IBillingViewAddModalState {
    isUploading: boolean;
    isLoading: boolean;
    error: Error | null;
    isForce: boolean;
    isActive: boolean;
    selectedWallet: string;
    selectedAccount: string;
    accountsOptions: IOptionInfo[];
    idAccountsOptions: any[];
}

export class BillingViewAddModal extends React.Component<IBillingViewAddModalProps, IBillingViewAddModalState> {
    state: IBillingViewAddModalState = {
        isUploading: false,
        isLoading: true,
        error: null,
        isForce: false,
        isActive: false,
        selectedAccount: '',
        selectedWallet: '',
        accountsOptions: [],
        idAccountsOptions: [],
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    addNewWallet() {
        const { reloadData, onClose, user_id } = this.props;
        const { isForce, isActive } = this.state;

        this.setState({
            isUploading: true,
        }, () => {
            const { selectedAccount, selectedWallet } = this.state;

            const query = selectedAccount ? 'account_id' : 'name';

            this.request.exec(REQUESTS.LINK_WALLET_USER, {
                body: {
                    action: 'link',
                    force: isForce,
                    active_flag: isActive,
                    user_id,
                    [query]: selectedAccount || selectedWallet,
                },
            })
                .then(() => {
                    reloadData();
                    onClose();
                })
                .catch((error) => {
                    this.setState({
                        isUploading: false,
                        error,
                    });

                    throw error;
                });
        });
    }

    fetchAccounts() {
        this.request.exec(REQUESTS.GET_WALLETS_DESCRIPTIONS)
            .then((response) => {
                const { accounts } = response;

                this.setState({
                    accountsOptions: accounts && accounts
                        .filter((account) => account.meta && account.meta.is_personal)
                        .map((account) => {
                            return {
                                text: account.meta && account.meta.hr_name || account.name,
                                value: account.name,
                            };
                        })
                        || [],
                    isLoading: false,
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                    isLoading: false,
                });
            });
    }

    fetchAccountDetails(selectedWallet, from = 0) {
        this.request.exec(REQUESTS.GET_ACCOUNTS, {
            queryParams: {
                account_name: encodeURIComponent(selectedWallet),
                from,
            },
        })
            .then((response) => {
                const { accounts, has_more } = response;
                const { idAccountsOptions } = this.state;

                this.setState({
                    error: null,
                    isLoading: has_more,
                    idAccountsOptions: accounts && [
                        ...idAccountsOptions,
                        ...accounts
                            .map((account) => {
                                return {
                                    text: account.data.meta && account.data.meta.comment || account.data.id,
                                    value: account.data.id,
                                };
                            })]
                        || [],
                }, () => {
                    if (has_more) {
                        this.fetchAccountDetails(selectedWallet, this.state.idAccountsOptions.length);
                    }
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                });
            });
    }

    componentDidMount() {
        this.fetchAccounts();
    }

    componentDidUpdate(prevProps, prevState: Readonly<IBillingViewAddModalState>) {
        if (this.state.selectedWallet !== prevState.selectedWallet && this.state.selectedWallet) {
            this.setState({ idAccountsOptions: [] },
                () => this.fetchAccountDetails(this.state.selectedWallet));
        }
    }

    componentWillUnmount() {
        this.request.abort();
    }

    onChange(key, value) {
        this.setState({
            [key]: value,
        } as Pick<IBillingViewAddModalState, 'isActive' | 'isForce' | 'selectedWallet' | 'selectedAccount'>);
    }

    render() {
        const { onClose } = this.props;
        const { isLoading, isUploading, error, isActive, isForce, idAccountsOptions, accountsOptions } = this.state;

        return (
            <Window onClose={onClose.bind(this)}
                    title={'Добавление кошелька'}
                    collapseError={error}>
                <Select options={accountsOptions}
                        initialValues={this.state.selectedWallet ? [this.state.selectedWallet] : null}
                        placeholder={'Тип кошелька'}
                        onSelect={this.onChange.bind(this, 'selectedWallet')}
                        required
                        addingOddVariants={isLoading}/>
                <Select options={idAccountsOptions}
                        placeholder={'ID аккаунта (не обязательно)'}
                        onSelect={this.onChange.bind(this, 'selectedAccount')}
                        addingOddVariants={isLoading}/>
                <div className={styles.container}>
                    <Checkbox checked={false}
                              onChange={this.onChange.bind(this, 'isActive')}/>
                    <span>Active: {isActive.toString()}</span>
                </div>
                <div className={styles.container}>
                    <Checkbox checked={false}
                              onChange={this.onChange.bind(this, 'isForce')}/>
                    <span>Force: {isForce.toString()}</span>
                </div>
                <div className={styles.container}>
                    <Button basic
                            onClick={onClose.bind(this)}>
                        Отмена
                    </Button>
                    <Button isLoading={isUploading}
                            onClick={this.addNewWallet.bind(this)}>Отправить</Button>
                </div>
                {isLoading
                    ? <div className={styles.loading_container}>
                        <Spin/>
                        <span>Данные кошельков загружаются</span>
                    </div> : null}
            </Window>
        );
    }
}
