import * as React from 'react';

import { EMPTY_DATA } from '../../../../constants';
import { useRequestHandler } from '../../../../hooks/useRequestHandler';
import { Collapse2 } from '../../../../ui/Collapse2';
import { NoInformation } from '../../../../ui/NoInformation';
import * as tblStyles from '../../../../ui/Table/index.css';
import { CSSettingsItems, CustomSettings } from '../../../../utils/customSettings';
import { Request2 } from '../../../../utils/request';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import { REQUESTS } from '../../request';

interface IB2bDetails {
    clientId: string;
    externalContractId: string;
    contractId: string;
    personId: string;
}

interface IBillingViewB2bDetailsProps {
    request: Request2;
    userId: string;
}

export const BillingViewB2bDetails = (props: IBillingViewB2bDetailsProps) => {
    const { request, userId } = props;
    const cs = new CustomSettings();

    return <Collapse2 title={'Данные корпоративного пользователя из Баланса'}
                      expandText={'Развернуть информацию'}
                      initialExpanded={cs.get(CSSettingsItems.userBalanceExpanded)}>
        <B2bDetailsTable request={request}
                         userId={userId}/>
    </Collapse2>;
};

interface IB2bDetailsTable {
    request: Request2;
    userId: string;
}

export const B2bDetailsTable = (props: IB2bDetailsTable) => {
    const { request, userId } = props;
    const options = React.useMemo(() => ({
        requestName: REQUESTS.GET_SESSIONS_CURRENT,
        requestOptions: {
            queryParams: {
                user_id: userId,
                traits: 'ReportB2BOrganization',
            },
        },
    }), [userId]);

    const requestHandler = (sessionCurrentData) => {
        return sessionCurrentData?.user?.billing?.b2b?.organizations.map((organization) => {
            const balanceInfo = organization?.details?.balance_info;

            return {
                personId: balanceInfo?.person_id ?? EMPTY_DATA,
                contractId: balanceInfo?.contract_id ?? EMPTY_DATA,
                clientId: balanceInfo?.client_id ?? EMPTY_DATA,
                externalContractId: balanceInfo?.external_contract_id ?? EMPTY_DATA,
            };
        });
    };

    const [
        isLoading,
        response,
        error,
        getB2BDetails,
    ] = useRequestHandler<IB2bDetails[]>(request, options, requestHandler);

    React.useEffect(() => {
        getB2BDetails();
    }, []);

    return <div>
        {error
            ? <SimpleError error={error}/>
            : isLoading
                ? <Spin/>
                : response?.length
                    ? <table className={`${tblStyles.table} ${tblStyles.clearFirstOpacity}`}>
                        <thead>
                            <tr>
                                <th>personId</th>
                                <th>contractId</th>
                                <th>clientId</th>
                                <th>externalContractId</th>
                            </tr>
                        </thead>
                        <tbody>
                            {response?.map((item, index) => {
                                return <tr key={index}>
                                    <td>{item.personId}</td>
                                    <td>{item.contractId}</td>
                                    <td>{item.clientId}</td>
                                    <td>{item.externalContractId}</td>
                                </tr>;
                            })}
                        </tbody>

                    </table>
                    : <NoInformation/>
        }
    </div>;
};
