import * as React from 'react';

import { Dict } from '../../../../types';
import { CUSTOM_TAGS } from '../../../constants';
import { TagRecord } from '../../../models/tag';
import { Button } from '../../../ui/Button';
import { Window } from '../../../ui/FullModal';
import Select, { IOptionInfo } from '../../../ui/Select';
import TextArea from '../../../ui/TextArea';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { LinksInput } from '../../BonusControlDialog';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import * as styles from './BillingViewAddModal.css';

interface IBillingViewCancelModalProps {
    payment: Dict<any>;
    userId: string;
    onClose: () => void;
    reloadData: () => void;
}

interface IBillingViewCancelModalState {
    [x: string]: any;

    isWorking: boolean;
    error: Error | null;
    allTags: any;
    tags: Dict<TagRecord>;
    sessions: any[];
    selectedSession: string[];
    selectedTag: string[];
    userComment: '';
    links: string[];
    isLoading: boolean;
}

const CANCEL_TAGS = [CUSTOM_TAGS.OPERATION_TAG, CUSTOM_TAGS.FIXED_SUM_TAG];
const BILLING_TAG_CANCEL = 'cancel';

export class BillingViewCancelModal extends
    React.Component<IBillingViewCancelModalProps, IBillingViewCancelModalState> {
    state: IBillingViewCancelModalState = {
        isWorking: false,
        error: null,
        allTags: {},
        sessions: [],
        links: [''],
        selectedSession: [],
        selectedTag: [''],
        userComment: '',
        tags: {},
        isLoading: false,
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    getData() {
        const { userId, payment } = this.props;

        const numdoc = 50;

        this.setState({
            isLoading: true,
        }, async () => {
            try {
                const response = await Promise.all([
                    this.request.exec(REQUESTS.GET_USER_SESSIONS, {
                        queryParams: {
                            user_id: userId,
                            numdoc,
                        },
                    }),
                    this.request.exec(REQUESTS.GET_TAG_DESCRIPTIONS),
                ]);

                const allTags = response[1]?.records?.reduce((p: any, c: any) => {
                    const meta = isValidJSONString(c.meta)
                        ? JSON.parse(c.meta)
                        : typeof(c.meta) === 'object' ? c.meta : {};

                    if (CANCEL_TAGS.includes(c.type) && meta.action === BILLING_TAG_CANCEL) {
                        return Object.assign({}, p, {
                            [c.name || '']: {
                                ...c,
                                meta,
                            },
                        });
                    }

                    return p;
                }, {}) || [];

                this.setState({
                    isLoading: false,
                    error: null,
                    allTags,
                    sessions: response[0].sessions
                        && Array.isArray(response[0].sessions)
                        && response[0].sessions
                            .sort((a: any, b: any) => {
                                const startA = a.segment && (a.segment.start || (a.segment?.meta?.start));
                                const startB = b.segment && (b.segment.start || (b.segment?.meta?.start));

                                return +startB - +startA;
                            })
                        || [],
                    selectedSession: payment.session_id || '',
                });
            } catch (error) {
                this.setState({
                    error,
                    isLoading: false,
                });
            }
        });
    }

    reasonsOptions(): IOptionInfo[] {
        const { allTags } = this.state;

        return Object.entries(allTags).map((el: any[]) => {
            return {
                text: el[1].display_name || el[1].name,
                value: el[0],
                description: el[1].comment,
            };
        });
    }

    onSelect(key, value: string) {
        if (value) {
            this.setState({ [key]: [value] });
        }
    }

    changeLink(index, value) {
        this.setState((prev) => {
            const { links } = prev;
            links.splice(index, 1, value);

            return {
                links,
            };
        });
    }

    deleteLink(index) {
        this.setState((prev) => {
            const { links } = prev;
            links.splice(index, 1);

            return {
                links,
            };
        });
    }

    onChange(param: string, value: any) {
        this.setState({
            [param]: value,
        });
    }

    cancelPayment() {
        const { onClose, userId, reloadData } = this.props;
        const { allTags, selectedTag, selectedSession, userComment, sessions } = this.state;

        const tag = allTags[selectedTag.toString()] || {};
        const session = sessions.filter((el: any) => el.segment
            && el.segment.meta && el.segment.meta.session_id == selectedSession)[0];
        const car = session && session.car || {};

        const links = this.state?.links
            ?.filter((uri) => uri !== '')
            ?.map((uri) => ({
                type: 'other',
                uri,
            })) ?? [];

        this.setState({
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.ATTACH_USER_TAG, {
                body: {
                    'car_number': car.number,
                    'comment': userComment,
                    'object_id': userId,
                    'priority': tag.default_priority && +tag.default_priority || 0,
                    'session_id': selectedSession.toString(),
                    'tag': selectedTag.toString(),
                    'links': links,
                },
            })
                .then(() => {
                    reloadData && reloadData();
                    onClose();
                })
                .catch((error) => {
                    this.setState({
                        isWorking: false,
                        error,
                    });
                });
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { onClose } = this.props;
        const { isWorking, error, selectedTag, userComment, links, isLoading } = this.state;

        return (
            <Window onClose={onClose.bind(this)}
                    title={'Отмена списания'}
                    error={error}>
                {isLoading
                    ? <Spin/>
                    : <>
                        <div>
                            <Select placeholder={'Причина'}
                                    options={this.reasonsOptions()}
                                    onSelect={this.onSelect.bind(this, 'selectedTag')}
                                    initialValues={selectedTag}/>
                            {
                                links?.map((link, index) => {
                                    return (
                                        <LinksInput key={index}
                                                    link={link}
                                                    onChange={this.changeLink.bind(this, index)}
                                                    onDelete={this.deleteLink.bind(this, index)}/>
                                    );
                                })
                            }
                            <TextArea placeholder={'Комментарий'}
                                      value={userComment}
                                      onChange={this.onChange.bind(this, 'userComment')}
                                      required>
                            </TextArea>
                        </div>
                        <div className={styles.container}>
                            <Button onClick={onClose.bind(this)}
                                    basic>Отмена</Button>
                            <Button isLoading={isWorking}
                                    onClick={this.cancelPayment.bind(this)}
                                    disabled={!userComment || !selectedTag[0]}>Отправить</Button>
                        </div>
                    </>
                }
            </Window>
        );
    }
}
