import * as React from 'react';

import { Dict } from '../../../../types';
import { UserInfoHandler } from '../../../models/user';
import { Collapse2 } from '../../../ui/Collapse2';
import { NoInformation } from '../../../ui/NoInformation';
import { UserStatus } from '../../../ui/Status';
import { Wallet } from '../../../ui/Wallet';
import { CSSettingsItems, CustomSettings } from '../../../utils/customSettings';
import { Request2 } from '../../../utils/request';
import { ITrustCard, IYandexAccount, PaymentSystem } from '../../Settings/Wallets/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { UserInfoContext } from '../context';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

interface IBillingViewCardsState {
    userTrustCards: Array<ITrustCard | IYandexAccount>;
    error: Error | null;
    cardsLoading: boolean;
    groupByUser: Dict<Array<ITrustCard | IYandexAccount>>;
}

export enum IBillingViewType {
    CARDS = 'cards',
    SAME_PERSONS = 'same_persons'
}

interface IBillingViewCardsProps {
    userId: string;
    type: IBillingViewType;
    additionalTitle?: string;
}

export class BillingViewCards extends React.Component<IBillingViewCardsProps, IBillingViewCardsState> {
    cs = new CustomSettings();

    state: IBillingViewCardsState = {
        userTrustCards: [],
        error: null,
        cardsLoading: true,
        groupByUser: {},
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    fetchCards() {
        const user_id = this.props.userId;

        this.request.exec(REQUESTS.GET_CARDS_TRUST, {
            queryParams: {
                user_id,
            },
        })
            .then((response) => {
                this.setState({
                    userTrustCards: response[this.props.type],
                    cardsLoading: false,
                    groupByUser: response?.[this.props.type]?.reduce((_p, _c) => {
                        const id = _c?.user?.id;
                        if (!_p.hasOwnProperty(id)) {
                            _p[id] = [_c];
                        } else {
                            _p[id].push(_c);
                        }

                        return _p;
                    }, {}) || [],
                });
            })
            .catch((walletsError) => {
                this.setState({
                    error: walletsError,
                    cardsLoading: false,
                });
            });
    }

    componentDidMount() {
        this.fetchCards();
    }

    componentDidUpdate(prevProps: Readonly<IBillingViewCardsProps>, prevState: Readonly<IBillingViewCardsState>) {
        if (this.props.userId !== prevProps.userId) {
            this.fetchCards();
        }
    }

    render() {
        const {
            cardsLoading,
            userTrustCards,
            error,
        } = this.state;

        const userUid = UserInfoHandler.getUid.call(this.context);
        const groupByUser = Object.entries(this.state.groupByUser);

        return <Collapse2 initialExpanded={this.cs.get(CSSettingsItems.cardsExpanded)}
                          expandText={`Развернуть способы оплаты ${this.props.additionalTitle || ''}`}
                          title={`Способы оплаты ${this.props.additionalTitle || ''}`}
                          className={`${styles.cardContainer} ${error || cardsLoading ? styles.displayBlock : ''}`}>
            {cardsLoading
                ? <Spin/>
                : error
                    ? <SimpleError error={error}/>
                    : !(userTrustCards && userTrustCards[0])
                        ? <NoInformation/>
                        : <>
                            {
                                groupByUser?.length
                                    ? groupByUser.map(el => {
                                        const user = el[1]?.[0]?.user;

                                        return <React.Fragment key={el[0]}>
                                            {user &&
                                            <p className={styles.user_info}>
                                                {UserInfoHandler.getPrintName.call(user)} (
                                                {UserInfoHandler.getUsername.call(user)})
                                                <UserStatus text={UserInfoHandler.getStatus.call(user)}/>
                                            </p>}

                                            {
                                                el[1]?.map((card, key) => {
                                                    const isYandexAccountOrCard = card
                                                        .payment_method === PaymentSystem.YANDEX_ACCOUNT
                                                        ? {
                                                            yandexAccount: card as IYandexAccount,
                                                        }
                                                        : {
                                                            card: card as ITrustCard,
                                                        };

                                                    return (
                                                        <Wallet scroogeLink={`https://scrooge.paysys.yandex-team.ru/binding_info/?creator_uid=${userUid}&user_account=${card.account}`}
                                                                key={key}
                                                                active
                                                                {...isYandexAccountOrCard}/>
                                                    );
                                                })
                                            }
                                        </React.Fragment>;

                                    })
                                    : <NoInformation/>}
                        </>
            }
        </Collapse2>;
    }
}

BillingViewCards.contextType = UserInfoContext;
