import * as React from 'react';

import { EMPTY_DATA, ONE_SECOND } from '../../../../../constants';
import { Button, ButtonTypes } from '../../../../../ui/Button';
import FormatDate from '../../../../../ui/FormatDate';
import { Link } from '../../../../../ui/Link';
import { rubs } from '../../../../../utils/rubs';
import { BillingViewCancelModal } from '../../BillingViewCancelModal';
import BillingViewForceModal from '../../BillingViewForceModal';
import * as styles from '../../index.css';
import TablePayments from '../../TablePayments';
import { IBillingCurrentTask } from '../../types';

interface IBillingCurrentTaskProps {
    task: IBillingCurrentTask;
    areAllExpanded: boolean;
    reloadData: () => void;
    userId: string;
    externalTooltipId: string;
    showNotAuthorized: boolean;
}

export const BillingCurrentTask = (props: IBillingCurrentTaskProps) => {
    const { task, areAllExpanded, reloadData, userId, externalTooltipId, showNotAuthorized } = props;
    const [isExpanded, setIsExpanded] = React.useState(false);
    const [isCancelModalShowing, setCancelModal] = React.useState(false);
    const [isForceModalShowing, setForceModal] = React.useState(false);

    const payments = task?.payments?.filter((payment) => {
        return showNotAuthorized ? true : payment.status !== 'not_authorized';
    }) ?? [];

    const showForceModal = (e) => {
        setForceModal(true);
        e.stopPropagation();
    };

    const showCancelModal = (e) => {
        setCancelModal(true);
        e.stopPropagation();
    };

    React.useEffect(() => {
        setIsExpanded(false);
    }, [areAllExpanded]);

    return (
        <>
            {isCancelModalShowing &&
            <BillingViewCancelModal reloadData={reloadData}
                                    onClose={setCancelModal.bind(null, false)}
                                    userId={task.user_id}
                                    payment={task}/>
            }

            {isForceModalShowing && (
                <BillingViewForceModal session_id={task.session_id}
                                       user_id={userId}
                                       reloadData={reloadData}
                                       onClose={setForceModal.bind(null, false)}/>
            )}
            <tr className={areAllExpanded || isExpanded ? styles.selectedRow : undefined}
                onClick={setIsExpanded.bind(null, !isExpanded)}>
                <td>
                    {payments?.length ? (
                        <span className={`${styles.expandToggle} ${areAllExpanded || isExpanded && styles.expandedToggle}`}>
                            {areAllExpanded || isExpanded ? 'Свернуть' : `Подробнее`}
                        </span>
                    ) : null}
                </td>
                <td>
                    <div className={`${styles.billing_type} ${styles[task.billing_type]}`}>
                        {task.billing_type || EMPTY_DATA}
                    </div>
                    <div><i>queue</i>: <strong>{task.queue}</strong></div>
                </td>
                <td>{rubs(task.bill) || EMPTY_DATA}</td>
                <td>{rubs(task.deposit) || EMPTY_DATA}</td>
                <td>
                    {task.session_id ?
                        <Link className={styles.statusLink}
                              href={`#/session/${task.session_id}?back=${encodeURIComponent(location.hash)}`}>
                            {task.task_status}
                        </Link>
                        : task.task_status}
                    {task.state || EMPTY_DATA}
                </td>
                <td><FormatDate value={task.last_status_update * ONE_SECOND}/></td>
                <td>
                    <Button className={styles.button}
                            onClick={showForceModal}
                            basic>Форсировать</Button>
                </td>
                <td>
                    {task.state === 'finished'
                        && <Button className={styles.button}
                                   onClick={showCancelModal}
                                   colorType={ButtonTypes.negative}
                                   basic>Отменить</Button>}
                </td>
            </tr>
            <TablePayments shouldShow={isExpanded || areAllExpanded}
                           externalTooltipId={externalTooltipId}
                           payments={payments}/>
        </>
    );
};
