import * as React from 'react';

import { UserInfoHandler } from '../../../../models/user';
import { Button } from '../../../../ui/Button';
import { Window } from '../../../../ui/FullModal';
import Select, { IOptionInfo } from '../../../../ui/Select';
import { IBlockRules } from '../../../../utils/IBlockRules';
import { Request2 } from '../../../../utils/request';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../../request';
import * as styles from '../BillingViewAddModal.css';

interface IBillingViewForceModalProps extends IBlockRules {
    user_id: string;
    session_id: string;
    onClose: () => void;
    reloadData: () => void;
}

interface IBillingViewForceModalState {
    isWorking: boolean;
    error: Error | null;
    cards: IOptionInfo[];
    same_persons_options: IOptionInfo[];
    clients_options: IOptionInfo[];
    card: string;
    same_persons_card: string;
    client: string;
    same_persons: any[];
}

const YANDEX_BANK_SYS = 'YandexBank';
const YANDEX_BANK_LAB = 'Счет в Яндексе';
const EXPIRED_LAB = '⚠️ КАРТА ПРОСРОЧЕНА ⚠️';

export class BillingViewForceModal extends React.Component<IBillingViewForceModalProps, IBillingViewForceModalState> {
    state: IBillingViewForceModalState = {
        isWorking: false,
        error: null,
        cards: [],
        same_persons_options: [],
        clients_options: [],
        card: '',
        same_persons_card: '',
        client: '',
        same_persons: [],
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    forcePayment() {
        const { same_persons_card, card, client } = this.state;
        const { reloadData, onClose, session_id, user_id } = this.props;
        const paymethod = same_persons_card || card || null;
        const debiting_user = client;

        const body = {
            ...(session_id ? { session_id } : { user_id }),
            ...paymethod && { paymethod },
            ...debiting_user && { debiting_user },
        };

        this.setState({
            isWorking: true,
        }, () => {
            this.request.exec(REQUESTS.FORCE_PAYMENT, { body })
                .then(() => {
                    reloadData();
                    onClose();
                })
                .catch((error) => {
                    this.setState({
                        isWorking: false,
                        error,
                    });
                });
        });
    }

    buildOptions(data) {
        return data && data?.map(card => {
            const expiredLabel = card.expired ? EXPIRED_LAB : null;
            let description = '';
            let text = card.account;

            if (card.user) {
                const printName = UserInfoHandler.getPrintName.call(card.user);
                const userName = UserInfoHandler.getUsername.call(card.user);

                description = `${printName}(${userName})`;
            }

            if (expiredLabel) {
                description += ` ${expiredLabel}`;
            }

            if (card.system === YANDEX_BANK_SYS) {
                text = YANDEX_BANK_LAB;
            }

            return { text, value: card.id, ...description && { description } };
        }) || [];
    }

    getData() {
        const { user_id } = this.props;

        this.request.exec(REQUESTS.GET_CARDS_TRUST, {
            queryParams: {
                user_id,
                cards_only: true,
            },
        })
            .then((response) => {
                const { cards, same_persons } = response;
                const cardsOption = this.buildOptions(cards);
                const same_persons_options = this.buildOptions(same_persons);

                const users: Record<string, any> = same_persons?.reduce((_p, _c) => {
                    if (!_p.hasOwnProperty(_c?.user?.id)) {
                        _p[_c.user.id] = [_c];
                    } else {
                        _p[_c.user.id].push(_c);
                    }

                    return _p;
                }, {}) || {};

                const clients_options = Object.entries(users)?.map(el => {
                    const user = el?.[1]?.[0]?.user;

                    return {
                        text: `${UserInfoHandler.getPrintName.call(user)} - ${UserInfoHandler.getUsername.call(user)}`,
                        value: el[0],
                        description: `status: ${UserInfoHandler.getStatus.call(user)}`,
                    };
                }) || [];

                this.setState({
                    cards: cardsOption,
                    same_persons_options,
                    clients_options,
                    same_persons,
                });
            });
    }

    onChangeCard(card) {
        this.setState({
            card,
            same_persons_card: '',
            client: '',
        });
    }

    onChangeSamePersonsCard(same_persons_card) {
        const { same_persons } = this.state;
        const obj = same_persons?.find(el => el.id === same_persons_card) || {};

        this.setState({
            same_persons_card,
            card: '',
            client: obj.user?.id || null,
        });
    }

    onChangeClient(client) {
        this.setState({
            card: '',
            same_persons_card: '',
            client,
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { onClose, BlockRules } = this.props;
        const {
            isWorking,
            error,
            cards,
            same_persons_options,
            clients_options,
            card,
            same_persons_card,
            client,
        } = this.state;

        return (
            <Window onClose={onClose.bind(this)}
                    title={'Перезапуск списания'}
                    error={error}>
                <Select initialValues={card && [card] || null}
                        placeholder={'Карта для списания'}
                        onSelect={this.onChangeCard.bind(this)}
                        options={cards}/>
                {BlockRules?.DebitSelectSamePersons && <>
                    <Select initialValues={same_persons_card && [same_persons_card] || null}
                            placeholder={'Карта для списания c дублей аккаунтов'}
                            onSelect={this.onChangeSamePersonsCard.bind(this)}
                            options={same_persons_options}/>
                    <Select initialValues={client && [client] || null}
                            placeholder={'Клиент-дубль'}
                            onSelect={this.onChangeClient.bind(this)}
                            options={clients_options}/>
                </>
                }

                <div className={styles.container}>
                    <Button onClick={onClose.bind(this)}
                            basic>Отмена</Button>
                    <Button isLoading={isWorking}
                            onClick={this.forcePayment.bind(this)}>Отправить</Button>
                </div>
            </Window>
        );
    }
}
