import React from 'react';
import { connect } from 'react-redux';

import { GVARS_INTERFACE_ADMIN_PREFIX } from '../../../../../constants';
import { SessionHistoryInfoHandler } from '../../../../../models/session';
import { Button } from '../../../../../ui/Button';
import { ErrorSign, ErrorsModal } from '../../../../../ui/ErrorBar';
import { getRawSetting2 } from '../../../../../utils/getRawSetting';
import { Request2 } from '../../../../../utils/request';
import { IStore } from '../../../../App/store';
import { REQUESTS, SESSION_REQUESTS } from '../../../../SessionCard/request';
import { BillingViewRefundModal } from '../../BillingViewRefundModal';
import style from '../../index.css';

interface IBonusButtonProps {
    sessionId: string;
    bonusLimit?: number;
}

interface IBonusButtonState {
    sessionsInfo: Record<string, any> | null;
    isModalOpened: boolean;
    isErrorModalOpen: boolean;
    isLoading: boolean;
    error: Error | null;
}

const PASS_OFFER = 'p2p_pass_offer';

class BonusButton extends React.Component<IBonusButtonProps, IBonusButtonState> {
    state: IBonusButtonState = {
        sessionsInfo: null,
        isModalOpened: false,
        isErrorModalOpen: false,
        isLoading: true,
        error: null,
    };
    request = new Request2({ requestConfigs: SESSION_REQUESTS });

    componentDidMount() {
        if (this.props.sessionId) {
            this.getSessionInfo();
        }
    }

    getSessionInfo() {
        this.request.exec(REQUESTS.GET_SESSION, {
            queryParams: {
                session_id: this.props.sessionId,
            },
        })
            .then(response => {
                this.setState({ sessionsInfo: response, isLoading: false });
            })
            .catch(error => {
                this.setState({ error, isLoading: false });
            });
    }

    openRefundModal(isModalOpened: boolean, e) {
        this.setState({ isModalOpened });
        isModalOpened && e.stopPropagation();
    }

    openErrorModal(isErrorModalOpen: boolean) {
        this.setState({ isErrorModalOpen });
    }

    render() {
        const { sessionsInfo = {}, isModalOpened, isLoading, error, isErrorModalOpen } = this.state;
        const { bonusLimit, sessionId } = this.props;
        const sih = SessionHistoryInfoHandler;

        const userId = sih.getUserId.call(sessionsInfo);
        const finalCost = sih.getFinalCost.call(sessionsInfo);
        const transferType = sih.getTransferredType.call(sessionsInfo);
        const delegationType = sih.getDelegationType.call(sessionsInfo);
        const isFinished = Boolean(sih.getFinish.call(sessionsInfo));
        const isNotDriving = !sih.getMileage.call(sessionsInfo);

        const isNotEvolution = sessionsInfo?.sessions?.length === 1;
        const isInLimit = finalCost < (bonusLimit ?? 0);
        const isTransferTypeCorrect = transferType !== 0;
        const isDelegationTypeCorrect = delegationType !== PASS_OFFER;

        const isValidSession = isInLimit
            && isFinished
            && isNotEvolution
            && isNotDriving
            && isTransferTypeCorrect
            && isDelegationTypeCorrect;

        const isDisabled = !sessionId || !sessionsInfo || !isValidSession;

        const DEFAULT_PAYMENT = {
            session_id: sessionId,
            sum: finalCost,
        };

        return <div className={style.buttonWrapper}>
            <Button basic={true}
                    className={style.bonus_button}
                    disabled={isDisabled}
                    isLoading={isLoading}
                    onClick={this.openRefundModal.bind(this, true)}>
                Возврат L1
            </Button>
            {error && (
                <ErrorSign onClick={this.openErrorModal.bind(this, true)}/>
            )}
            {error && isErrorModalOpen
                ? <ErrorsModal onClose={() => this.openErrorModal(false)}
                               errors={[error]}/>
                : null
            }
            {isModalOpened && userId
                ? <BillingViewRefundModal onClose={this.openRefundModal.bind(this, false)}
                                          userId={userId}
                                          amount={finalCost}
                                          isL1={true}
                                          payment={DEFAULT_PAYMENT}/>
                : null
            }
        </div>;
    }
}

const mapStateToProps = (store: IStore) => {
    const key = `${GVARS_INTERFACE_ADMIN_PREFIX}.bonus_limit`;

    return {
        bonusLimit: getRawSetting2(store, key),
    };
};

export default connect(mapStateToProps)(BonusButton);
