import * as React from 'react';

import { Dict } from '../../../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../../../constants';
import { TagRecordHandler } from '../../../../../models/tag';
import { Button } from '../../../../../ui/Button';
import FormatDate from '../../../../../ui/FormatDate';
import { Link } from '../../../../../ui/Link';
import { rubs } from '../../../../../utils/rubs';
import { BillingViewRefundModal } from '../../BillingViewRefundModal';
import * as styles from '../../index.css';
import TablePayments from '../../TablePayments';
import { BillingType, IBillingHistoryTask, ITags } from '../../types';
import BonusButton from './BonusButton';

interface BillingHistoryTaskProps {
    task: IBillingHistoryTask;
    areAllExpanded: boolean;
    reloadData: () => void;
    userId: string;
    tags?: ITags;
    externalTooltipId: string;
    BlockRules?: Dict<boolean>;
    showNotAuthorized?: boolean;
    index: number;
}

export const BillingHistoryTask = (props: BillingHistoryTaskProps) => {
    const {
        task,
        areAllExpanded,
        reloadData,
        tags = {},
        showNotAuthorized = false,
        externalTooltipId,
        BlockRules,
        index,
    } = props;
    const [isExpanded, setIsExpanded] = React.useState(false);
    const [isRefundModalShowing, setRefundModal] = React.useState(false);

    const payments = task.payments && task.payments.filter((payment) => {
        return showNotAuthorized ? true : payment.status !== 'not_authorized';
    });

    const tagLinks = TagRecordHandler.getTagLinks.call(tags?.[task?.session_id]?.add)?.[0];
    let rulingNumberGibdd = null;

    if (task.billing_type === BillingType.gibdd) {
        try {
            rulingNumberGibdd =
                JSON.parse(TagRecordHandler.getTagComment.call(tags?.[task?.session_id]?.add)).ruling_number;
        } catch {

        }
    }

    const showRefundModal = (e) => {
        setRefundModal(true);
        e.stopPropagation();
    };

    const expand = () => {
        setIsExpanded(!isExpanded);
    };

    const DEFAULT_PAYMENT = {
        session_id: task?.session_id,
        sum: task?.bill,
    };

    React.useEffect(() => {
        setIsExpanded(false);
    }, [areAllExpanded]);

    return (
        <>
            {isRefundModalShowing && props.userId && (
                <BillingViewRefundModal amount={0}
                                        onClose={setRefundModal.bind(null, false)}
                                        userId={props.userId}
                                        payment={task?.payments?.[0] ?? DEFAULT_PAYMENT}
                                        reloadData={reloadData}/>
            )}
            <tr className={areAllExpanded || isExpanded ? styles.selectedRow : undefined}
                onClick={expand}>
                <td>{index + 1}</td>
                <td className={styles.nowrap}>
                    {payments?.length ? (
                        <span className={`${styles.expandToggle} ${areAllExpanded || isExpanded && styles.expandedToggle}`}>
                            {areAllExpanded || isExpanded ? 'Свернуть' : `Подробнее (${payments.length}) `}
                        </span>
                    ) : null}
                </td>
                <td>
                    <div>
                        {BlockRules?.Refound ?
                            <Button className={styles.button}
                                    onClick={showRefundModal}
                                    basic>
                                Вернуть
                            </Button>
                            : null
                        }
                    </div>
                    <div>
                        {BlockRules?.ShowButtonRefundGeneral
                            ? <BonusButton sessionId={task?.session_id ?? ''}/>
                            : null
                        }
                    </div>
                </td>
                <td>
                    <div className={`${styles.billing_type} ${styles[task.billing_type]}`}>
                        {task.billing_type || EMPTY_DATA}
                    </div>
                    <div>
                        {task.session_id && tags?.hasOwnProperty(task.session_id)
                            ? tagLinks
                                ? <Link href={tagLinks}>
                                    {TagRecordHandler.getTagDisplayName.call(tags?.[task.session_id]?.add)}
                                </Link>
                                : TagRecordHandler.getTagDisplayName.call(tags?.[task.session_id]?.add)
                            : null
                        }
                        {rulingNumberGibdd
                            ? `Пост. ${rulingNumberGibdd}`
                            : null
                        }
                    </div>
                </td>
                <td className={styles.nowrap}>{rubs(task.bill) || EMPTY_DATA}</td>
                <td>
                    {task.session_id && ![BillingType.ticket, BillingType.gibdd].includes(task.billing_type)
                        ? <Link className={styles.statusLink}
                                href={`#/session/${task?.meta?.real_session_id ?? task.session_id}/billing?back=${encodeURIComponent(location.hash)}`}>
                            сессия
                        </Link>
                        : EMPTY_DATA
                    }
                </td>
                <td colSpan={2}>
                    <FormatDate withSecond
                                value={task.timestamp * ONE_SECOND}/>
                </td>
            </tr>
            <TablePayments shouldShow={isExpanded || areAllExpanded}
                           externalTooltipId={externalTooltipId}
                           payments={payments}/>
        </>
    );
};
