import React from 'react';
import ReactTooltip from 'react-tooltip';

import { Dict, TAG_ACTION } from '../../../../types';
import { COINS_PER_RUBLE, EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { TagRecord, TagRecordHandler } from '../../../models/tag';
import { Button } from '../../../ui/Button';
import FormatDate, { FormatDateInString } from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import Select, { IOptionInfo } from '../../../ui/Select';
import * as tblStyles from '../../../ui/Table/index.css';
import TextArea from '../../../ui/TextArea';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { LinksInput } from '../../BonusControlDialog';
import * as styles from '../../Settings/Wallets/AddModal.css';
import { DEFAULT_TAG_REFUND, REFUND_TAG, SCROOGEMCDUCK } from '../../Settings/Wallets/types';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';

interface IBillingViewRefundModalState {
    [key: string]: any;

    isLoading: boolean;
    isUploading: boolean;
    onChangeData: any;
    error: Error | null;
    allTags: any;
    tags: Dict<TagRecord>;
    sessions: any[];
    selectedSession: string[];
    selectedTag: string[];
    userComment: '';
    compensation: number;
    amount: number;
}

interface IBillingViewRefundModalProps {
    amount: number;
    userId: string;
    payment: Dict<any>;
    onClose: () => void;
    reloadData?: () => void;
    isL1?: boolean;
}

export class BillingViewRefundModal extends
    React.Component<IBillingViewRefundModalProps, IBillingViewRefundModalState> {
    state: IBillingViewRefundModalState = {
        isLoading: true,
        isUploading: false,
        tags: {},
        allTags: {},
        onChangeData: undefined,
        error: null,
        sessions: [],
        links: [''],
        selectedSession: [],
        selectedTag: [DEFAULT_TAG_REFUND],
        amount: this.props?.amount / COINS_PER_RUBLE,
        userComment: '',
        compensation: 0,
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    onSelectTag(value: string) {
        if (value) {
            this.setState({ selectedTag: [value] });
        }
    }

    onSelectSession(value: string) {
        if (value) {
            this.setState({ selectedSession: [value] });
        }
    }

    onChange(param: string, e: any) {
        let value = e;

        if (['compensation', 'amount'].includes(param)) {
            value = value.replace(',', '.');
        }

        this.setState({
            [param]: value,
        });
    }

    reasonsOptions(): IOptionInfo[] {
        const { allTags } = this.state;
        const { isL1 } = this.props;

        const L1_TAG_PREFIX = 'refund_general';

        const tagsArray = isL1
            ? Object.entries(allTags)?.filter(el => {
                return el[0]?.includes(L1_TAG_PREFIX);
            })
            : Object.entries(allTags);

        return tagsArray.map((el: any[]) => {
            return {
                text: el[1].display_name || el[1].name,
                value: el[0],
                description: el[1].comment,
            };
        });
    }

    ridesOptions(): IOptionInfo[] {
        const { sessions } = this.state;

        return (
            [{ text: EMPTY_DATA, value: '' }].concat(sessions?.map((el: any) => {
                const car = el.car && el.car.number || EMPTY_DATA;
                const { session_id = '', start = 0, finish = 0 } = el.segment
                && el.segment.meta
                || el.segment;

                return {
                    text: `${FormatDateInString({
                        value: start * ONE_SECOND,
                        onlyDate: true,
                    })} ${FormatDateInString({
                        value: start * ONE_SECOND,
                        onlyTime: true,
                    })} - ${FormatDateInString({
                        value: finish * ONE_SECOND,
                        onlyTime: true,
                    })}, ${car}`,
                    value: session_id,
                    description: '',
                };
            }))
        );
    }

    addLink() {
        this.setState((prev) => ({
            links: [...prev.links, ''],
        }));
    }

    changeLink(index, value) {
        this.setState((prev) => {
            const { links } = prev;
            links.splice(index, 1, value);

            return {
                links,
            };
        });
    }

    deleteLink(index) {
        this.setState((prev) => {
            const { links } = prev;
            links.splice(index, 1);

            return {
                links,
            };
        });
    }

    getData() {
        const { userId, payment } = this.props;
        const numdoc = 50;
        const TAG_DESCRIPTION_RESPONSE = 2;

        this.setState({
            isLoading: true,
        }, async () => {
            try {
                const response = await Promise.all([
                    this.request.exec(REQUESTS.GET_USER_SESSIONS, {
                        queryParams: {
                            user_id: userId,
                            numdoc,
                        },
                    }),
                    this.request.exec(REQUESTS.GET_TAGS_HISTORY, {
                        queryParams: {
                            object_id: userId,
                            duration: '90d',
                        },
                    }),
                    this.request.exec(REQUESTS.GET_TAG_DESCRIPTIONS),
                ]);

                const refund_tags = response[TAG_DESCRIPTION_RESPONSE].records
                    && response[TAG_DESCRIPTION_RESPONSE].records.filter((tag) => tag.type === REFUND_TAG) || [];

                const refund_tags_names = refund_tags.map((tag) => tag.name);

                const tags = response[1].records
                    && response[1].records.reduce((tags: Dict<TagRecord>, tag: TagRecord) => {
                        if (refund_tags_names.includes(TagRecordHandler.getTagName.call(tag))) {
                            tags[TagRecordHandler.getTagId.call(tag)] = {
                                ...tags[TagRecordHandler.getTagId.call(tag)],
                                [TagRecordHandler.getAction.call(tag)]: tag,
                            };
                        }

                        return tags;
                    }, {}) || {};

                const allTags = refund_tags.reduce((p: any, c: any) => {
                    let meta = c.meta;
                    if (meta) {
                        meta = isValidJSONString(meta)
                            ? JSON.parse(meta)
                            : typeof(meta) === 'object' ? meta : {};
                    }

                    return Object.assign({}, p, {
                        [c.name || '']: {
                            ...c,
                            meta,
                        },
                    });
                }, {}) || [];

                this.setState({
                    isLoading: false,
                    error: null,
                    tags,
                    allTags,
                    sessions: response[0].sessions
                        && Array.isArray(response[0].sessions)
                        && response[0].sessions
                            .sort((a: any, b: any) => {
                                const startA = a.segment
                                    && (a.segment.start || (a.segment.meta && a.segment.meta.start));
                                const startB = b.segment
                                    && (b.segment.start || (b.segment.meta && b.segment.meta.start));

                                return +startB - +startA;
                            })
                        || [],
                    selectedSession: payment.session_id || '',
                });
            } catch (error) {
                this.setState({
                    error,
                    isLoading: false,
                });
            }
        });
    }

    uploadData() {
        const { onClose, userId, reloadData } = this.props;
        const { allTags, selectedTag, selectedSession, userComment, sessions, compensation, amount } = this.state;

        const tag = allTags[selectedTag.toString()] || {};
        const session = sessions.filter((el: any) => el.segment
            && el.segment.meta && el.segment.meta.session_id == selectedSession)[0];
        const car = session && session.car || {};

        const links = this.state?.links
            ?.filter((uri) => uri !== '')
            ?.map((uri) => ({
                type: 'other',
                uri,
            })) ?? [];

        this.setState({
            isUploading: true,
        }, () => {
            this.request.exec(REQUESTS.ATTACH_USER_TAG, {
                body: {
                    'car_number': car.number,
                    'comment': userComment,
                    'object_id': userId,
                    'priority': tag.default_priority && +tag.default_priority || 0,
                    'session_id': selectedSession.toString(),
                    'tag': selectedTag.toString(),
                    'links': links,
                    'compensation': +compensation || 0,
                    'amount': +(+amount * COINS_PER_RUBLE).toFixed(0) || 0,
                },
            })
                .then(() => {
                    reloadData && reloadData();
                    onClose();
                })
                .catch((error) => {
                    this.setState({
                        isUploading: false,
                        error,
                    });
                });
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const {
            isLoading,
            error,
            allTags,
            selectedSession,
            selectedTag,
            userComment,
            /*    compensation,*/
            amount,
            tags,
            links,
        } = this.state;

        const { onClose } = this.props;

        return (
            <Window onClose={onClose.bind(this)}
                    title={'Возвраты'}
                    error={error}>
                <div className={styles.accountDetailsContainer}>
                    {isLoading ? <Spin/> : (
                        <>
                            <div className={styles.accountContainerForm}>
                                <div>
                                    <Select placeholder={'Причина'}
                                            options={this.reasonsOptions()}
                                            onSelect={this.onSelectTag.bind(this)}/>
                                    <p>{allTags[selectedTag.toString()] &&
                                    allTags[selectedTag.toString()].comment || EMPTY_DATA}</p>
                                    <Select placeholder={'Поездка'}
                                            options={this.ridesOptions()}
                                            onSelect={this.onSelectSession.bind(this)}
                                            initialValues={selectedSession}/>
                                    {
                                        links?.map((link, index) => {
                                            return (
                                                <LinksInput key={index}
                                                            link={link}
                                                            onChange={this.changeLink.bind(this, index)}
                                                            onDelete={this.deleteLink.bind(this, index)}/>
                                            );
                                        })
                                    }
                                    <TextArea placeholder={'Комментарий'}
                                              value={userComment}
                                              onChange={this.onChange.bind(this, 'userComment')}
                                              required>
                                    </TextArea>
                                    {/* <Input
                                        value={compensation}
                                        placeholder={'Компенсация бонусами'}
                                        onChange={this.onChange.bind(this, 'compensation')}/>*/}

                                    <Input value={amount}
                                           placeholder={'Сумма (руб.)'}
                                           onChange={this.onChange.bind(this, 'amount')}/>
                                </div>
                                <div className={styles.buttonContainer}>
                                    <Button onClick={onClose.bind(this)}
                                            basic>Отмена</Button>
                                    <Button isLoading={isLoading}
                                            onClick={this.uploadData.bind(this)}
                                            disabled={!userComment}>Отправить</Button>
                                </div>
                            </div>
                            <div className={styles.accountDetailsHistory}>
                                <h3>История</h3>
                                <table className={tblStyles.table}>
                                    <tbody>
                                        {Object.entries(tags)
                                            .filter(el => {
                                                return TagRecordHandler.getTagUsername.call(el[1][TAG_ACTION.REMOVE])
                                                === SCROOGEMCDUCK;
                                            })
                                            .map((el, index) => {
                                                const item = el[1][TAG_ACTION.REMOVE] || {};
                                                const item_0 = el[1][TAG_ACTION.ADD] || {};

                                                const amount = item.tag_details && item.tag_details.amount || 0;

                                                return (
                                                    <tr key={item.tag_id}>
                                                        <td>{index + 1}</td>
                                                        <td>
                                                            <FormatDate value={item.timestamp * ONE_SECOND}
                                                                        withSecond={true}/>
                                                        </td>
                                                        <td>{TagRecordHandler.getTagUsername.call(item_0)}</td>
                                                        <td>
                                                            <span data-tip
                                                                  data-for={item.tag_id}>
                                                                {rubs(amount)}
                                                            </span>
                                                        </td>
                                                        <ReactTooltip id={item.tag_id}
                                                                      aria-haspopup='true'
                                                                      place="top"
                                                                      type="info">
                                                            {item.tag_display_name}
                                                        </ReactTooltip>
                                                    </tr>
                                                );
                                            })
                                        }
                                    </tbody>
                                </table>
                            </div>
                        </>
                    )}
                </div>
            </Window>
        );
    }
}
