import * as React from 'react';

import { Dict } from '../../../../types';
import { ONE_SECOND } from '../../../constants';
import { TagRecordHandler } from '../../../models/tag';
import FormatDate from '../../../ui/FormatDate';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import * as tblStyle from '../../../ui/Table/index.css';
import { isObjectEqual } from '../../../utils/isObjectEqual';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { REFUND_TAG } from '../../Settings/Wallets/types';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

interface IBillingViewRefundTableProps {
    timelapse?: string;
    tags: Dict<any>;
    sessionId?: string;
    tagsError: Error | null;
}

interface IBillingViewRefundTableState {
    isLoading: boolean;
    refundsHistory: Dict<any>;
    error: Error | null;
}

export class BillingViewRefundTable extends
    React.Component<IBillingViewRefundTableProps, IBillingViewRefundTableState> {
    state: IBillingViewRefundTableState = {
        isLoading: true,
        refundsHistory: {},
        error: null,
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    getTags() {
        this.setState({
            isLoading: true,
        }, async () => {
            try {
                const response = await this.request.exec(REQUESTS.GET_TAG_DESCRIPTIONS);

                const refundTags = response.records && response.records
                    .filter((tag: any) => tag.type === REFUND_TAG)
                    .map((tag: any) => tag.name);

                const refundsHistory = this.props.tags?.records?.reduce((p: any, c: any) => {
                    if (refundTags.includes(c.tag_name)) {
                        const hasId = p.hasOwnProperty(c.tag_id);
                        const isSession = (this.props.sessionId
                            ? this.props.sessionId === c.tag_details?.session_id
                            : true);

                        if (!hasId && isSession) {
                            p[c.tag_id] = {
                                items: [c],
                                tag_name: c.tag_display_name || c.tag_name,
                                comment: c.tag_details?.comment,
                                amount: c.tag_details?.amount,
                            };
                        } else {
                            p[c.tag_id]?.items?.push(c);
                        }
                    }

                    return p;
                }, {});

                this.setState({
                    refundsHistory,
                    isLoading: false,
                });
            } catch (error) {
                this.setState({
                    error,
                    isLoading: false,
                });
            }
        });
    }

    componentDidMount() {
        this.getTags();
    }

    componentDidUpdate(prevProps: Readonly<IBillingViewRefundTableProps>) {
        if (!isObjectEqual(this.props.tags, prevProps.tags)) {
            this.getTags();
        }
    }

    render() {
        const { refundsHistory, isLoading, error } = this.state;
        const { timelapse = '', tagsError } = this.props;

        return <>
            <h4>История возвратов {timelapse}</h4>
            {error || tagsError
                ? <SimpleError error={error || tagsError}/>
                : isLoading
                    ? <Spin/>
                    : <>
                        {!(refundsHistory && Object.keys(refundsHistory).length) ? <NoInformation/> : (
                            <table className={`${tblStyle.table} ${tblStyle.clearBackground} 
                            ${tblStyle.clearFirstOpacity} ${styles.table}`}>
                                <thead>
                                    <tr>
                                        <th>Тег</th>
                                        <th>Комментарий</th>
                                        <th>Cумма</th>
                                        <th>История</th>
                                        <th>Ссылки</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {Object.entries(refundsHistory).map((item: any, key: number) => {
                                        return (
                                            <tr key={`${key}_main`}>
                                                <td>{item[1].tag_name}</td>
                                                <td>{item[1].comment}</td>
                                                <td>
                                                    <span>{rubs(item[1].amount)}</span>
                                                </td>
                                                <td>
                                                    {item[1].items.map((item: any, key: number) => {
                                                        return (
                                                            <div key={`${key}_main`}>
                                                                <span>{key + 1})</span>
                                                                <span>
                                                                    <FormatDate value={item.timestamp * ONE_SECOND}
                                                                                withSecond={true}/>
                                                                </span>
                                                                <span>{item.action}</span>
                                                                <span>
                                                                    {item.user_data_full && (
                                                                        <Link href={`#/clients/${item.user_data_full.id}/info`}>
                                                                            {item.user_data_full.username}
                                                                        </Link>
                                                                    )}
                                                                </span>
                                                            </div>
                                                        );
                                                    })}
                                                </td>
                                                <td>
                                                    {
                                                        TagRecordHandler.getTagLinks.call(item[1]?.items[0])
                                                            .map((uri, index) => {
                                                                return <Link key={`link-${index}`}
                                                                             href={uri}
                                                                             target={'_blank'}>
                                                                    {uri}
                                                                </Link>;
                                                            })
                                                    }
                                                </td>
                                            </tr>
                                        );
                                    })
                                    }
                                </tbody>
                            </table>
                        )}
                    </>
            }
        </>;
    }
}
