import moment from 'moment';
import * as React from 'react';

import { Dict } from '../../../../types';
import { EMPTY_DATA, ONE_DAY, ONE_SECOND, ONE_WEEK_IN_DAYS } from '../../../constants';
import DatePicker from '../../../ui/DatePicker';
import Select from '../../../ui/Select';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import { BillingViewAccounts } from './BillingViewAccounts';
import BillingViewB2bDetails from './BillingViewB2bDetails';
import { BillingViewCards, IBillingViewType } from './BillingViewCards';
import BillingViewCurrent from './BillingViewCurrent';
import BillingViewHistory from './BillingViewHistory';
import { BillingViewRefundTable } from './BillingViewRefundTable';
import * as styles from './index.css';

interface IBillingViewProps {
    userId: string;
    BlockRules: Dict<boolean>;
}

interface IBillingViewState {
    [key: number]: any;

    error: Error | null;
    since: number;
    until: number;
    isBillingInfoLoading: boolean;
    currentFilter: string;
    billingInfo: Dict<any>;
    timeError: boolean;
    tags: Dict<any>;
    tagsError: Error | null;
}

const BILLING_FILTERS = [
    { text: 'Всё', value: '' },
    { text: 'Поездка', value: 'car_usage', description: 'car_usage' },
    { text: 'Штраф', value: 'ticket', description: 'ticket' },
    { text: 'ГИБДД', value: 'ticket_gibdd', description: 'ticket_gibdd' },
    { text: 'Топливо', value: 'fueling', description: 'fueling' },
    { text: 'Сервис', value: 'service', description: 'service' },
];

const MAX_DATE_MONTH_NUMBER = 32;

export class BillingView extends React.Component<IBillingViewProps, IBillingViewState> {
    state: IBillingViewState = {
        since: Date.now() - ONE_WEEK_IN_DAYS * ONE_DAY,
        until: Date.now(),
        isBillingInfoLoading: true,
        currentFilter: '',
        billingInfo: {},
        error: null,
        timeError: false,
        tags: {},
        tagsError: null,
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    onChange(key, value) {
        this.setState({
            [key]: value,
        } as Pick<IBillingViewState, 'currentFilter'>);
    }

    fetchBillingInfo() {
        const { userId } = this.props;
        const { since, until } = this.state;

        this.request.exec(REQUESTS.GET_BILLING_INFO, {
            queryParams: {
                user_id: userId,
                since: Math.round(since / ONE_SECOND),
                until: Math.round(until / ONE_SECOND),
            },
        })
            .then((response) => {
                this.setState({
                    billingInfo: response,
                    isBillingInfoLoading: false,
                });

                this.getTags();
            })
            .catch((error) => {
                this.setState({
                    error,
                    isBillingInfoLoading: false,
                });
            });
    }

    getTags() {
        const { userId } = this.props;
        const { since, until } = this.state;

        this.request.exec(REQUESTS.GET_TAGS_HISTORY, {
            queryParams: {
                object_id: userId,
                since: Math.round(since / ONE_SECOND),
                until: Math.round(until / ONE_SECOND),
            },
        })
            .then((response) => {
                this.setState({
                    tags: response,
                });
            })
            .catch((tagsError) => {
                this.setState({
                    tagsError,
                });
            });
    }

    modifyDate(key, value) {
        this.setState({
            timeError: false,
        });

        this.setState({
            [key]: +value,
        });
    }

    componentDidMount() {
        this.fetchBillingInfo();
    }

    shouldComponentUpdate(nextProps: Readonly<IBillingViewProps>, nextState: Readonly<IBillingViewState>): boolean {
        return !Number.isNaN(nextState.since)
            && !Number.isNaN(nextState.until)
            && ((nextState.since > MAX_DATE_MONTH_NUMBER || nextState.since === 0)
                && (nextState.until > MAX_DATE_MONTH_NUMBER || nextState.until === 0));
    }

    componentDidUpdate(prevProps: Readonly<IBillingViewProps>, prevState: Readonly<IBillingViewState>) {
        const { since, until } = this.state;

        if (this.props.userId !== prevProps.userId
            || (since !== prevState.since)
            || (until !== prevState.until)) {
            if (since < until && since && until) {
                this.fetchBillingInfo();
            } else {
                this.setState({
                    timeError: true,
                });
            }
        }
    }

    render() {
        const {
            since,
            until,
            billingInfo,
            currentFilter,
            tags,
            error,
            isBillingInfoLoading,
            timeError,
            tagsError,
        } = this.state;

        const { userId, BlockRules } = this.props;

        const history_data = billingInfo?.compiled_history?.sort((a, b) => b?.timestamp - a?.timestamp);

        moment.locale('ru');

        const timelapse = `c ${since ? moment(since).format('D MMMM') : EMPTY_DATA}
            по ${until ? moment(until).format('D MMMM') : EMPTY_DATA}`;

        return (
            <>
                {error ? <SimpleError error={error}/> : isBillingInfoLoading ? <Spin/> : (
                    <div className={styles.userBillingContainer}>
                        <BillingViewAccounts userId={userId}/>
                        <BillingViewCards userId={userId} type={IBillingViewType.CARDS}/>
                        <BillingViewCards userId={userId}
                                          type={IBillingViewType.SAME_PERSONS}
                                          additionalTitle={'(с дублей)'}/>
                        {BlockRules?.UserBalance && <>
                            <BillingViewB2bDetails request={this.request}
                                                   userId={this.props.userId}/>
                        </>}
                        <div className={styles.filtersContainer}>
                            <Select options={BILLING_FILTERS}
                                    onSelect={this.onChange.bind(this, 'currentFilter')}
                                    placeholder={'Фильтр по типу списания'}/>
                            <DatePicker value={since}
                                        placeholder={'C'}
                                        onChange={this.modifyDate.bind(this, 'since')}/>
                            <DatePicker value={until}
                                        placeholder={'По'}
                                        onChange={this.modifyDate.bind(this, 'until')}/>
                            {timeError && (
                                <p className={styles.timeError}>Проверьте правильность введенных дат!</p>
                            )}
                        </div>
                        <BillingViewCurrent userId={userId}
                                            data={billingInfo && billingInfo.current}
                                            currentFilter={currentFilter}
                                            reloadData={this.fetchBillingInfo.bind(this)}/>
                        <hr className={styles.separator}/>
                        {/*DRIVEFRONT-1074*/}
                        <BillingViewCurrent additionalTitle={'(c дублей)'}
                                            userId={userId}
                                            data={billingInfo && billingInfo.same_persons}
                                            currentFilter={currentFilter}
                                            reloadData={this.fetchBillingInfo.bind(this)}/>

                        <hr className={styles.separator}/>
                        <BillingViewHistory tags={tags}
                                            userId={userId}
                                            data={history_data}
                                            timelapse={timelapse}
                                            currentFilter={currentFilter}
                                            reloadData={this.fetchBillingInfo.bind(this)}/>
                        <hr className={styles.separator}/>
                        <BillingViewRefundTable tagsError={tagsError}
                                                tags={tags}
                                                timelapse={timelapse}/>
                    </div>
                )}
            </>
        );
    }
}
