import moment from 'moment';
import * as React from 'react';
import { connect } from 'react-redux';

import { ISearchControls, WithSearchControls } from '../../../../decorators/SearchControls';
import { Dict } from '../../../../types';
import { TagRecord, TagRecordHandler } from '../../../models/tag';
import FormatDate from '../../../ui/FormatDate';
import { NoInformation } from '../../../ui/NoInformation';
import * as tblStyle from '../../../ui/Table/index.css';
import { ConstantsKey, fetchConstants } from '../../../utils/fetchConstants';
import { isValidJSONString } from '../../../utils/isValidJSONString';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { BonusLinks } from '../../BonusControlDialog';
import CarNumber from '../../CarNumber';
import { IAccount, ROBOT } from '../../Settings/Wallets/types';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

interface Internal extends ISearchControls {
    constants: Promise<any>;
}

const mapDispatchToProps = (dispatch: any) => {
    return {
        constants: fetchConstants([ConstantsKey.IFACE_TAG_FILTERS], dispatch),
    };
};

interface IBonusViewHistoryProps {
    userId: string;
    bonusAccounts: IAccount[];
}

interface IBonusViewHistoryState {
    isLoading: boolean;
    tags: TagRecord[];
    allTags: Dict<string>[];
}

const SIXTY_DAYS = 60;

@WithSearchControls({ className: styles.bonusHistory, since: moment(new Date()).subtract(SIXTY_DAYS, 'd') })
class BonusViewHistoryInternal extends React.Component<IBonusViewHistoryProps & Internal, IBonusViewHistoryState> {
    state = {
        tags: [],
        isLoading: false,
        allTags: [],
    };
    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    getData() {
        const { userId, since, until } = this.props;
        const TAGS_INDEX = 2;

        this.setState({
            isLoading: true,
        }, () => {
            Promise.all([
                this.props.constants,
                this.request.exec(REQUESTS.GET_TAGS_HISTORY,{
                    queryParams: {
                        object_id: userId,
                        since: since,
                        until: until,
                    },
                }),
                this.request.exec(REQUESTS.GET_TAG_DESCRIPTIONS),
            ])
                .then(response => {
                    const constBonus: string[] = [];

                    this.props.bonusAccounts?.forEach((account) => {
                        if (response[0]?.iface_tag_filters[account.type_name]) {
                            constBonus.push(...response[0]?.iface_tag_filters[account.type_name]);
                        }
                    });

                    const tags = response[1].records
                        && response[1].records.reduce((tags: TagRecord[], tag: TagRecord) => {
                            if (constBonus.includes(TagRecordHandler.getTagName.call(tag))) {
                                const tagId = TagRecordHandler.getTagId.call(tag);

                                tags[tagId] = {
                                    ...tags[tagId],
                                    [TagRecordHandler.getAction.call(tag)]: tag,
                                };
                            }

                            return tags;
                        }, {}) || {};

                    const allTags = response[TAGS_INDEX].records
                        && response[TAGS_INDEX].records.reduce((_p: any, _c: any) => {
                            let _meta = _c.meta;
                            if (_meta) {
                                _meta = isValidJSONString(_meta)
                                    ? JSON.parse(_meta)
                                    : typeof(_meta) === 'object' ? _meta : {};
                            }

                            return {
                                ..._p,
                                [_c.name || '']: {
                                    ..._c,
                                    _meta,
                                },
                            };
                        }, {}) || [];

                    this.setState({
                        tags,
                        isLoading: false,
                        allTags,
                    });
                });
        });
    }

    componentDidUpdate(prevProps: Readonly<any>): void {
        if (prevProps.since !== this.props.since
            || prevProps.until !== this.props.until
            || prevProps.userId !== this.props.userId
            || prevProps.bonusAccounts !== this.props.bonusAccounts) {
            this.getData();
        }
    }

    render() {
        const { isLoading, tags, allTags } = this.state;
        const tagsEntries = Object.entries(tags) || [];

        return (
            <div className={styles.bonusSection}>
                {isLoading ? <Spin/> :
                    !tagsEntries.length ? <NoInformation /> : (
                        <>
                            <BonusViewHistoryTable data={tagsEntries}
                                                   allTags={allTags}/>
                        </>
                    )}
            </div>
        );
    }
}

interface IBonusViewHistoryTableProps {
    data: TagRecord[];
    allTags: any;
}

const BonusViewHistoryTable = (props: IBonusViewHistoryTableProps) => {
    const { data, allTags } = props;
    const [isExpanded, setExpanded] = React.useState(false);

    const FIRST_HISTORY_COUNT = 3;

    const firstTags = data.slice(0, FIRST_HISTORY_COUNT);

    const expand = () => {
        setExpanded(!isExpanded);
    };

    return (
        <table className={`${tblStyle.table} ${tblStyle.clearAllBackground} ${tblStyle.clearFirstOpacity}` +
            ` ${styles.table}`}>
            <thead>
                <tr>
                    <th>#</th>
                    <th>Дата</th>
                    <th />
                    <th>Пользователь</th>
                    <th>Тег</th>
                    <th>Машина</th>
                    {/*<th>Ссылки</th>*/}
                    <th>Комментарий</th>
                </tr>
            </thead>
            <tbody>
                {(isExpanded || data.length <= FIRST_HISTORY_COUNT) ? (
                    <>
                        {data.map((item: IBillingTagEntries, index: number) => {
                            return (
                                <BonusViewHistoryItem item={item}
                                                      allTags={allTags}
                                                      index={index}
                                                      key={index}/>
                            );
                        })}
                        {isExpanded ? (
                            <tr>
                                <td className={styles.expandToggle}
                                    colSpan={7}
                                    onClick={expand}>
                                    Cвернуть
                                </td>
                            </tr>
                        ) : undefined}
                    </>
                ) : (
                    <>
                        {firstTags.map((item: IBillingTagEntries, index: number) => {
                            return (
                                <BonusViewHistoryItem item={item}
                                                      allTags={allTags}
                                                      index={index}
                                                      key={index}/>
                            );
                        })}
                        <tr>
                            <td className={styles.expandToggle}
                                colSpan={7}
                                onClick={expand}>
                                Развернуть ({data.length - FIRST_HISTORY_COUNT})
                            </td>
                        </tr>
                    </>
                )}
            </tbody>
        </table>
    );
};

type IBillingTagEntries = [string, {
    [key: string]: TagRecord;
}]

interface IBillingHistoryItem {
    item: IBillingTagEntries;
    allTags: any;
    index: number;
}

const BonusViewHistoryItem = (props: IBillingHistoryItem) => {
    const { item, allTags, index } = props;
    const { remove = {} as TagRecord, add = {} as TagRecord } = item[1];
    const removeUser = TagRecordHandler.getTagUsername.call(remove);

    const addTagName = TagRecordHandler.getTagName.call(add);
    const bonusLinks = TagRecordHandler.getTagLinks.call(add);
    const amountDeadline = TagRecordHandler.getTagAmountDeadline.call(add);

    let { action = '', amount = 0 } = allTags[addTagName]
        && allTags[addTagName]._meta
        && allTags[addTagName]._meta || {};

    if (!amount) {
        amount = TagRecordHandler.getTagAmount.call(add);
    }

    const classNameTr = `${styles.bonusHistoryItem} ${ROBOT === removeUser ? styles[`action_${action}`] : ''}`;

    return (
        <>
            <tr className={classNameTr}>
                <td>{index + 1}</td>
                <td>
                    <FormatDate className={styles.tagDate}
                                value={TagRecordHandler.getTagTimestamp.call(add)}
                                withSecond/>
                    <FormatDate className={styles.tagDate}
                                value={TagRecordHandler.getTagTimestamp.call(remove)}
                                withSecond/>
                </td>
                <td>
                    <strong>{action === 'credit' ? '+' : '-'}{rubs(amount)}</strong>
                    {amountDeadline && (
                        <div className={styles.limited_balances}>
                            до <FormatDate value={amountDeadline}/>
                        </div>
                    )}
                </td>
                <td>
                    <div>add: {TagRecordHandler.getTagUsername.call(add)}</div>
                    <div>rem: {removeUser} </div>
                </td>
                <td>
                    <div>{TagRecordHandler.getTagDisplayName.call(add)}</div>
                    <div>
                        {allTags[addTagName]
                            && allTags[addTagName].comment}
                    </div>
                </td>
                <td>
                    {TagRecordHandler.getTagCarNumber.call(add) ? (
                        <CarNumber value={TagRecordHandler.getTagCarNumber.call(add)}/>
                    ) : undefined}
                </td>
                <td>
                    {TagRecordHandler.getTagComment.call(add)}
                    <div className={styles.smallComment}>
                        {item[0]}
                    </div>
                </td>
            </tr>
            {bonusLinks.length
                ? <BonusLinks links={bonusLinks}
                              colSpan={5}
                              className={classNameTr}/>
                : null
            }
        </>
    );
};

export const BonusViewHistory = connect(() => {
    return {};
}, mapDispatchToProps)(BonusViewHistoryInternal);
