import * as React from 'react';

import { TAG_ACTION } from '../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../constants';
import { UserInfoHandler } from '../../../models/user';
import { Button } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { Window } from '../../../ui/FullModal';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import * as tblStyle from '../../../ui/Table/index.css';
import { Request2 } from '../../../utils/request';
import { RESPONSE_STATUSES, SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';

export interface IPromo {
    activity_deadline: string;
    activity_start: string;
    code: string;
    id: string;
    prefix: string;
    remaining: number;
    removing_deadline: string;
    type: string;
    usage_limit: string;
    given_out: string;
    history?: any[];
    description: string;
    generator: string;
}

interface IBonusViewPromoProps {
    userId: string;
}

interface IBonusViewPromoState {
    isLoading: boolean;
    codes: IPromo[];
    error: Error | null;
    showSearchModal: boolean;
}

export class BonusViewPromo extends React.Component<IBonusViewPromoProps, IBonusViewPromoState> {
    state: IBonusViewPromoState = {
        isLoading: true,
        codes: [],
        error: null,
        showSearchModal: false,
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    getData() {
        const { userId } = this.props;

        this.request.exec(REQUESTS.GET_PROMO_HISTORY, {
            queryParams: {
                user_id: userId,
            },
        })
            .then((response) => {
                this.setState({
                    codes: response.codes,
                    isLoading: false,
                });
            })
            .catch((error) => {
                this.setState({
                    error,
                    isLoading: false,
                });
            });
    }

    toggleSearch() {
        this.setState({
            showSearchModal: !this.state.showSearchModal,
        });
    }

    componentDidMount() {
        this.getData();
    }

    render() {
        const { isLoading, error, codes, showSearchModal } = this.state;

        return (
            <>
                {showSearchModal && (
                    <SearchPromoModal onClose={this.toggleSearch.bind(this)}/>
                )}
                <h4>История промокодов (<Link onClick={this.toggleSearch.bind(this)}>поиск по коду</Link>)</h4>
                <PromoTable codes={codes}
                            error={error ? <SimpleError error={error} /> : null}
                            isLoading={isLoading}/>
            </>
        );
    }
}

interface ISearchPromoModalProps {
    onClose: () => void;
}

interface ISearchPromoModalState {
    codes: IPromo[];
    isLoading: boolean;
    error: Error | null;
    searchCode: string;
}

export class SearchPromoModal extends React.Component<ISearchPromoModalProps, ISearchPromoModalState> {
    state: ISearchPromoModalState = {
        codes: [],
        isLoading: false,
        error: null,
        searchCode: '',
    };
    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    onChange(searchCode: string) {
        this.setState({
            searchCode,
        });
    }

    getData() {
        const { searchCode } = this.state;

        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_PROMO_HISTORY_BY_CODE, {
                queryParams: {
                    code: searchCode,
                },
            })
                .then((response) => {
                    this.setState({
                        codes: response.codes,
                        isLoading: false,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    render() {
        const { onClose } = this.props;
        const { codes, isLoading, error } = this.state;

        return (
            <Window onClose={onClose.bind(this)}
                    title={'Промокоды'}>
                <div className={styles.searchPromo}>
                    <Input value={this.state.searchCode}
                           onChange={this.onChange.bind(this)}
                           placeholder={'Поиск'}
                           className={styles.inputPromo}/>
                    <Button onClick={this.getData.bind(this)}
                            className={styles.buttonPromo}>Найти</Button>
                </div>
                <PromoTable codes={codes}
                            isLoading={isLoading}
                            error={error ? <CustomError error={error}/> : null}/>
            </Window>
        );
    }
}

const CustomError = (props: { error: any }) => {
    const {
        http_code = 0,
        ui_message = '',
        ui_title = '',
    } = props.error && props.error.data && props.error.data.error_details || {};
    if (http_code === RESPONSE_STATUSES.CODE_404) {
        return (
            <div className={styles.customError}>
                <div>ui_title: <strong>{ui_title}</strong></div>
                <div>ui_message: <strong>{ui_message}</strong></div>
            </div>
        );
    }

    return <SimpleError error={props.error} data={{ label: 'Информация о промокодах' }}/>;

};

interface IPromoTableProps {
    codes: IPromo[];
    error: any;
    isLoading: boolean;
    settings?: boolean;
    sortByGenerator?: boolean;
}

export const PromoTable = React.memo((props: IPromoTableProps) => {
    const { codes, error, isLoading, settings, sortByGenerator } = props;
    const [isSortingByGenerator, shouldSortByGenerator] = React.useState(false);

    const immutable_codes = [...codes];

    if (isSortingByGenerator) {
        immutable_codes.sort((a, b) => {
            return a.generator.localeCompare(b.generator);
        });
    }

    const sort = () => {
        shouldSortByGenerator(!isSortingByGenerator);
    };

    return (
        <>
            {isLoading ? <Spin /> : error ? error :
                !immutable_codes.length ? <NoInformation /> : (
                    <table className={`${tblStyle.table} ${tblStyle.clearBackground}`}>
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Префикс</th>
                                <th>Использование</th>
                                <th>Начало</th>
                                <th>Дедлайн</th>
                                {settings ? (
                                    <th>Дедлайн на удаление</th>
                                ) : undefined}
                                <th>Отдали</th>
                                <th>
                                Генератор {sortByGenerator && (
                                        <Link onClick={sort}>(отсортировать)</Link>
                                    )}
                                </th>
                                <th>Описание</th>
                            </tr>
                        </thead>
                        <tbody>
                            {immutable_codes.map((code, index) => {
                                return (
                                    <PromoRow key={index}
                                              code={code}
                                              index={index}
                                              settings={settings}/>
                                );
                            })}
                        </tbody>
                    </table>
                )}
        </>
    );
});

const PromoRow = React.memo((props: { code: IPromo; index: number; settings?: boolean}) => {
    const { code, index, settings } = props;

    const used = code.hasOwnProperty('usage_limit') && code.hasOwnProperty('remaining')
        ? (+code.usage_limit - +code.remaining) : EMPTY_DATA;

    return (
        <>
            <tr key={code.id}>
                <td>{index + 1}</td>
                <td>{code.prefix}</td>
                <td>{used} / {code.usage_limit}</td>
                <td>
                    <FormatDate value={+code.activity_start * ONE_SECOND}
                                withSecond/>
                </td>
                <td>
                    <FormatDate value={+code.activity_deadline * ONE_SECOND}
                                withSecond/>
                </td>
                {settings ? (
                    <td>
                        <FormatDate value={+code.removing_deadline * ONE_SECOND}
                                    withSecond/>
                    </td>
                ) : undefined}
                <td>{code.given_out || EMPTY_DATA}</td>
                <td>{code.generator || EMPTY_DATA}</td>
                <td>{code.description || EMPTY_DATA}</td>
            </tr>
            {code.hasOwnProperty('history') ? (
                code.history && code.history.map((el: any, index: number) => {
                    const historyUsername = UserInfoHandler.getPrintNameWithId.call(el?.user_data_full);
                    const historyUserId = el.user_data_full && el.user_data_full.id;

                    return (
                        <tr key={index}
                            className={styles.promoHistory}>
                            <td colSpan={2}/>
                            <td>
                                <FormatDate value={el.timestamp * ONE_SECOND}
                                            withSecond/>
                            </td>
                            <td>
                                <strong>
                                    {el.action === TAG_ACTION.ADD ? 'Заведен промокод через админку'
                                        : 'Активирован промокод'}
                                </strong>
                            </td>
                            <td>{el.remaining} / {code.usage_limit}</td>
                            <td>
                                <Link href={`#/clients/${historyUserId}/info`}>{historyUsername}</Link>
                            </td>
                            <td colSpan={2} />
                        </tr>
                    );
                })
            ) : undefined}

        </>
    );
});
