import * as React from 'react';

import { ONE_SECOND } from '../../../constants';
import { Button } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { IBlockRules } from '../../../utils/IBlockRules';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import { buttonLocationDetails, buttonNameDetails } from '../../../utils/sendLogs/eventTypes/buttonDetails';
import { BonusControlDialog } from '../../BonusControlDialog';
import { AccountType, IAccount } from '../../Settings/Wallets/types';
import { CLIENTS_CARD_REQUESTS, REQUESTS } from '../request';
import { BonusViewHistory } from './BonusViewHistory';
import { BonusViewPromo } from './BonusViewPromo';
import * as styles from './index.css';

interface IBonusViewProps extends IBlockRules {
    userId: string;
}

const GENERAL_TYPENAME = 'bonus';

interface IBonusViewState {
    isLoading: boolean;
    error: Error | null;
    accounts: IAccount[];
    showManageBonuses: boolean;
}

export class BonusView extends React.Component<IBonusViewProps, IBonusViewState> {
    state: IBonusViewState = {
        isLoading: false,
        error: null,
        accounts: [],
        showManageBonuses: false,
    };

    request = new Request2({ requestConfigs: CLIENTS_CARD_REQUESTS });

    getData() {
        const { userId } = this.props;

        this.setState({
            isLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_ACCOUNTS, {
                queryParams: {
                    user_id: userId,
                },
            })
                .then(response => {
                    this.setState({
                        accounts: response.accounts
                            && response.accounts.filter((account: IAccount) => account.type === AccountType.BONUS)
                            || [],
                        isLoading: false,
                        error: null,
                    });
                })
                .catch((error) => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    componentDidMount() {
        this.getData();
    }

    showBonusModal(showManageBonuses) {
        this.setState({
            showManageBonuses,
        });
    }

    closeModal(showManageBonuses) {
        this.showBonusModal(showManageBonuses);
        this.getData();
    }

    render() {
        const { accounts, showManageBonuses } = this.state;
        const { userId } = this.props;
        const additionalAccounts: JSX.Element[] = [];
        const limitedBalance: any = [];

        let balance = 0;
        let spent = 0;

        {
            accounts.forEach((account, index) => {
                if (account.is_actual) {
                    balance += account.balance;
                }

                if (account.details.limited_balance) {
                    limitedBalance.push(...account.details.limited_balance);
                }

                if (account.type_name !== GENERAL_TYPENAME) {
                    additionalAccounts.push(
                        <div key={`add-account${index}`}
                             className={`${styles.headerBalance} ${styles.additionalAccount}`}>
                            <h3 className={!account?.is_actual ? styles.disabled_account : ''}>
                                {account.name}:
                            </h3>
                            <div className={`${styles.balances} ${!account?.is_actual ? styles.disabled_account : ''}`}>
                                <p>накоплено: <span className={styles.sum}>
                                    {rubs(account?.balance)}
                                </span></p>
                                <p>потрачено: <span className={styles.spent}>{rubs(account?.spent)}</span></p>
                            </div>
                        </div>,
                    );
                }

                spent += account.spent;
            });
        }

        return (
            <>
                {showManageBonuses && (
                    <BonusControlDialog onClose={this.closeModal.bind(this, false)}
                                        userId={userId}/>
                )}
                <div className={`${styles.header} ${styles.bonusSection}`}>
                    <div className={styles.headerBalance}>
                        <h3>Бонусы:</h3>
                        <div className={styles.balances}>
                            <p>
                                всего накоплено: <span className={styles.sum}>{rubs(balance)}</span>
                                <div className={styles.limited_balances_container}>
                                    {limitedBalance.map((balance, index) => {
                                        return <p className={styles.limited_balances}
                                                  key={index}>
                                            {rubs(balance.sum)} до <FormatDate value={balance.deadline * ONE_SECOND}/>
                                            {index < limitedBalance.length - 1 ? ',' : ''}
                                        </p>;
                                    })}
                                </div>
                            </p>
                            <p>всего потрачено: <span className={styles.spent}>{rubs(spent)}</span></p>
                        </div>
                    </div>
                    {
                        this.props.BlockRules
                        && this.props.BlockRules['bonus_button']
                        && <Button onClick={this.showBonusModal.bind(this, true)}
                                   ytLog={{
                                       button_name: buttonNameDetails.BONUS_BUTTON,
                                       location: buttonLocationDetails.CLIENT_CARD,
                                   }}>
                            Управление бонусами
                        </Button>
                    }
                </div>
                {additionalAccounts.length ? (
                    <div className={`${styles.balancesSection} ${styles.bonusSection} `}>
                        {additionalAccounts}
                    </div>
                ) : undefined}
                <BonusViewHistory userId={userId}
                                  bonusAccounts={accounts}/>
                <BonusViewPromo userId={userId}/>
            </>
        );
    }
}
