import * as React from 'react';

import { ICar } from '../../../../../types';
import { FREE_DELEGATION } from '../../../../constants';
import { CAR_STATUS, CarInfoHandler } from '../../../../models/car';
import SessionsHistoryItem, { SessionHistoryInfoHandler } from '../../../../models/session';
import UserInfo from '../../../../models/user';
import { DelegationIcon } from '../../../../ui/DelegationIcon';
import { Link } from '../../../../ui/Link';
import CarNumber from '../../../CarNumber';
import { copySessionText } from '../../../Chats/utils';
import { Copy } from '../../../Copy';
import * as styles from './index.css';

interface ICCSessionInfoProps {
    sessions: SessionsHistoryItem | null;
    index: number;
    car?: ICar | null;
    user: UserInfo | null;
}

export const CCSessionInfo = (props: ICCSessionInfoProps) => {
    const { sessions, car = null, user, index } = props;
    const isFinished = SessionHistoryInfoHandler.getFinish.call(sessions, index);
    const carModel = SessionHistoryInfoHandler.getModelName.call(sessions, index);
    const carStatus = CarInfoHandler.getStatus.call(car);
    const isReserved = car && [CAR_STATUS.old_state_reserv, CAR_STATUS.old_state_reserv_testing]
        .includes(carStatus);
    const isDelegated = car?.tags?.find(el => el.tag === FREE_DELEGATION) || false;

    const copyInfo = () => {
        const carNumber = SessionHistoryInfoHandler.getCarNumber.call(sessions, index);

        return copySessionText(user as UserInfo, carNumber);
    };

    const back = encodeURIComponent(location.hash);
    let cardColor = `${styles.session_info} ${isFinished ? styles.finished : ''}`;
    switch (carStatus) {
    case CAR_STATUS.old_state_reserv:
    case CAR_STATUS.old_state_reserv_testing:
        cardColor += ` ${styles.reserved}`;
        break;
    case CAR_STATUS.old_state_accept:
        cardColor += ` ${styles.accepted}`;
        break;
    case CAR_STATUS.old_state_parkin:
        cardColor += ` ${styles.parking}`;
        break;
    default:
        break;
    }

    return (
        <>
            <div className={cardColor}>
                <div className={styles.car_info}>
                    <CarNumber carInfo={SessionHistoryInfoHandler.getCar.call(sessions, index)}/>
                    <Link className={styles.car_link}
                          href={`#/cars/${SessionHistoryInfoHandler.getCarId.call(sessions, index)}/info?back=${back}`}>
                        {carModel}
                        {isDelegated && <DelegationIcon/>}
                    </Link>
                </div>
                <Copy internalTooltipId={`session_info_${index}`}
                      className={styles.copy}
                      text={copyInfo()}>
                    <div>
                        <p className={styles.status}>
                            {car && CarInfoHandler.getStatus.call(car)}
                        </p>
                        <div>
                            <h4>{isFinished ? 'ПОСЛЕДНЯЯ ПОЕЗДКА' : isReserved ? 'ОЖИДАНИЕ' : 'В ПОЕЗДКЕ'}</h4>
                            <div className={styles.additional_info}>
                                <span>
                                    {SessionHistoryInfoHandler.getDuration.call(sessions, index)}
                                </span>
                                <span>
                                    {SessionHistoryInfoHandler.getHrMileage.call(sessions, index)}
                                </span>
                            </div>
                        </div>
                    </div>
                </Copy>
            </div>
        </>
    );
};
