import * as React from 'react';

import { Dict, ICar } from '../../../../../types';
import { ONE_SECOND } from '../../../../constants';
import { CAR_ACTIONS, CarInfoHandler, IConfig } from '../../../../models/car';
import SessionsHistoryItem, { SessionHistoryInfoHandler } from '../../../../models/session';
import UserInfo, { UserInfoHandler } from '../../../../models/user';
import { Button, ButtonTypes } from '../../../../ui/Button';
import FormatDate from '../../../../ui/FormatDate';
import { QuickActionButton } from '../../../../ui/QuickActionButton';
import LS from '../../../../utils/localStorage/localStorage';
import { Request2 } from '../../../../utils/request';
import { buttonLocationDetails, buttonNameDetails } from '../../../../utils/sendLogs/eventTypes/buttonDetails';
import { openNewTab } from '../../../../utils/utils';
import CreateChatWindow from '../../../CreateChatWindow';
import { CHAT_TYPE } from '../../../CreateChatWindow/types';
import FastTagsView from '../../../FastTagsView';
import TagModal from '../../../TagModal';
import { ITag, OBJECT_TYPES } from '../../../TagModal/component';
import { CC_REQUESTS, REQUESTS } from '../request';
import * as styles from './index.css';
import { SidebarButtons } from './SidebarButtons';

const CAR_SERVICE_TAG = 'car_service_tag';

interface ICCSidebarProps {
    sessions: SessionsHistoryItem | null;
    index: number;
    user: UserInfo | null;
    car: ICar | null;
    BlockRules: Dict<boolean>;
    getData: () => void;
    transformToScooter: (key: string) => string;
}

interface ICCSidebarState {
    question: string;
    confirmIsOpen: boolean;
    accept: () => void;
    isWorking: boolean;
    password: string;
    tagType: any[];
    isTagModalOpen: boolean;
    fastTag: ITag | undefined;
    error: Error | null;
    showCreateChat: boolean;
    showCreatCall: boolean;
}

export class CCSidebar extends React.Component<ICCSidebarProps, ICCSidebarState> {
    state: ICCSidebarState = {
        question: '',
        confirmIsOpen: false,
        accept: () => {},
        isWorking: false,
        password: '',
        isTagModalOpen: false,
        tagType: [],
        fastTag: undefined,
        error: null,
        showCreateChat: false,
        showCreatCall: false,
    };

    request = new Request2({ requestConfigs: CC_REQUESTS });
    ls = new LS();

    defaultConfig = {
        finishOrder: false,
        callback: () => {},
        ignoreError: false,
        question: '',
    };

    onClose() {
        this.setState({
            confirmIsOpen: false,
            isWorking: false,
        });
    }

    carControlHandle(actionType: string, config: IConfig = this.defaultConfig) {
        const { car } = this.props;
        const { car_id } = config;

        const accept = this.handleCarAction.bind(this, config.actionName === REQUESTS.FINISH_CAR_SESSION
            ? config.user_id
            : car_id || car && car.id,
        actionType, config);

        return accept();
    }

    handleCarAction(id: string, type: string, config: IConfig = this.defaultConfig): Promise<any> {
        const {
            callback, actionName = REQUESTS.CAR_ACTION_STAFF
            , evolution_mode = 'default',
        } = config;
        const { sessions, index } = this.props;

        const headers = actionName === REQUESTS.FINISH_CAR_SESSION ? { UserIdDelegation: id } : {};
        const session_id = SessionHistoryInfoHandler.getSessionId.call(sessions, index);

        return this.request.exec(actionName, {
            headers: headers,
            queryParams: { command: type, car_id: id, evolution_mode, session_id },
        })
            .then(() => {
                if (callback) {
                    callback();
                }
            });
    }

    openTagModal(tagType: string[]) {
        this.setState({
            tagType,
            isTagModalOpen: true,
        });
    }

    addTag(fastTag: ITag) {
        this.setState({
            isTagModalOpen: true,
            fastTag,
        });
    }

    closeTagModal() {
        this.setState({ isTagModalOpen: false });
    }

    fos() {
        const { user, sessions, index } = this.props;
        const userName = UserInfoHandler.getName.call(user);
        const fullUserName = UserInfoHandler.getPrintName.call(user);
        const number = SessionHistoryInfoHandler.getCarNumber.call(sessions, index);
        const phone = UserInfoHandler.getSimplePhone.call(user);
        const email = UserInfoHandler.getEmail.call(user);
        const href = `https://forms.yandex-team.ru/surveys/17084/?user_name=${encodeURIComponent(userName)}`
            + `&full_user_name=${encodeURIComponent(fullUserName)}`
            + `&user_phone=${encodeURIComponent(phone)}&user_email=${email}&auto_id=${encodeURIComponent(number)}`;

        openNewTab(href);
    }

    showCreatChat(showCreateChat) {
        this.setState({
            showCreateChat,
        });
    }

    showCreatCall(showCreatCall) {
        this.setState({
            showCreatCall,
        });
    }

    render() {
        let {} = this.state;
        const { sessions, index, user, car, BlockRules } = this.props;
        const sessionMeta = CarInfoHandler.getSessionMeta.call(car);
        const manageHood = BlockRules?.DutySupport || car?.model_id !== 'skoda_rapid';
        const sessionConfirm = sessionMeta
            ? `${FormatDate({
                value: sessionMeta.start * ONE_SECOND,
                withSecond: true,
            }).props.children} - ${FormatDate({
                value: sessionMeta.finish * ONE_SECOND,
                withSecond: true,
            }).props.children}`
            : null;
        const questionDetails = <div>
            <div>Авто: <strong>{car && car.number}</strong>{car && car.model_id}</div>
            <div>Пользователь: <strong>{UserInfoHandler.getPhone.call(user)}</strong>{name}</div>
            <div>Заказ: {sessionConfirm}</div>
        </div>;

        return (
            <div className={styles.sidebar}>
                {this.state.isTagModalOpen
                    ? <TagModal objectId={{ type: OBJECT_TYPES.CAR, id: car && car.id || undefined }}
                                tagType={this.state.tagType}
                                onClose={this.closeTagModal.bind(this)}
                                initialData={this.state.fastTag}/>
                    : null
                }
                {sessions && !SessionHistoryInfoHandler.getFinish.call(sessions, index) && (
                    <QuickActionButton question={<>Завершить заказ? {questionDetails}</>}
                                       button={{
                                           children: 'Завершить поездку',
                                           colorType: ButtonTypes.warning,
                                       }}
                                       showConfirm
                                       accept={this.carControlHandle.bind(this, CAR_ACTIONS.ORDERFINISH, {
                                           actionName: REQUESTS.FINISH_CAR_SESSION,
                                           user_id: UserInfoHandler.getId.call(user),
                                           evolution_mode: 'default',
                                           callback: this.props.getData,
                                       })}/>
                )}
                <Button className={styles.button}
                        onClick={this.fos.bind(this)}
                        ytLog={{ button_name: buttonNameDetails.FOS, location: buttonLocationDetails.CLIENT_CARD }}>
                    ФОС
                </Button>
                {sessions
                    && <Button className={styles.button}
                               colorType={ButtonTypes.positive}
                               onClick={this.openTagModal.bind(this, [CAR_SERVICE_TAG])}
                               ytLog={{
                                   button_name: buttonNameDetails.SERVICE_TAG,
                                   location: buttonLocationDetails.CLIENT_CARD,
                               }}>
                        Сервисная задача
                    </Button>}

                {sessions && <FastTagsView place={buttonLocationDetails.CLIENT_CARD}
                                           addTag={this.addTag.bind(this)}/>
                }
                {
                    this.props?.BlockRules?.OutgoingChat
                        ? <Button colorType={ButtonTypes.positive}
                                  className={styles.button}
                                  onClick={this.showCreatChat.bind(this, true)}
                                  ytLog={{
                                      button_name: buttonNameDetails.CREATE_CHAT,
                                      location: buttonLocationDetails.CLIENT_CARD,
                                  }}>Создать чат</Button>
                        : null
                }
                {
                    this.props?.BlockRules?.OutgoingCall
                        ? <Button colorType={ButtonTypes.positive}
                                  className={styles.button}
                                  onClick={this.showCreatCall.bind(this, true)}
                                  ytLog={{
                                      button_name: buttonNameDetails.CREATE_CALL,
                                      location: buttonLocationDetails.CLIENT_CARD,
                                  }}>Отдать на прозвон</Button>
                        : null
                }

                {
                    this.state.showCreateChat
                    && <CreateChatWindow onClose={this.showCreatChat.bind(this, false)}
                                         type={CHAT_TYPE.chats}
                                         object_id={UserInfoHandler.getId.call(user)}/>
                }
                {
                    this.state.showCreatCall
                    && <CreateChatWindow onClose={this.showCreatCall.bind(this, false)}
                                         type={CHAT_TYPE.outgoing}
                                         object_id={UserInfoHandler.getId.call(user)}/>
                }

                {sessions
                    ? <SidebarButtons BlockRules={BlockRules}
                                      manageHood={manageHood}
                                      carControlHandle={this.carControlHandle.bind(this)}
                                      questionDetails={questionDetails}/>
                    : null
                }
            </div>
        );
    }
}
