import * as React from 'react';

import { ITagDescriptionsPropsExport, TagDescriptions } from '../../../../decorators/TagDescriptions';
import { ICar } from '../../../../types';
import SessionsHistoryItem, { ISessionInfo, SessionHistoryInfoHandler } from '../../../models/session';
import UserInfo from '../../../models/user';
import { Request2 } from '../../../utils/request';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import CCViewEmptySession from './CCViewEmptySession';
import CCViewSession from './CCViewSession';
import * as styles from './index.css';
import { CC_REQUESTS, REQUESTS } from './request';

interface ICCViewProps extends ITagDescriptionsPropsExport {
    userId: string;
}

interface ICCViewState {
    error: Error | null;
    isLoading: boolean;
    user: UserInfo | null;
    currentSessions: SessionsHistoryItem | null;
    cars: ICar[];
    sessionCurrentData: any;
    carIds: string[];
}

const CAR_INFO_TIMEOUT = 3000;
const RETRY_LIMIT = 5;

@TagDescriptions()
export class CCView extends React.Component<ICCViewProps, ICCViewState> {
    state: ICCViewState = {
        error: null,
        isLoading: false,
        user: null,
        currentSessions: null,
        cars: [],
        sessionCurrentData: {},
        carIds: [],
    };

    limit = 0;
    timer: any;

    request = new Request2({ requestConfigs: CC_REQUESTS });

    getSession(i, currentSessions, carIds, allSessions) {
        currentSessions.push(allSessions.sessions[i]);
        carIds.push(SessionHistoryInfoHandler.getCarId.call(allSessions, i));
    }

    componentWillUnmount() {
        clearTimeout(this.timer);
    }

    update() {
        clearTimeout(this.timer);
        this.timer = setTimeout(async () => {
            let cars: ICar[];
            const carRequests: Promise<ICar>[] = this.state.carIds.map((car_id) => {
                return this.request.exec(REQUESTS.GET_CAR_INFO, {
                    queryParams: {
                        car_id,
                    },
                });
            });

            try {
                cars = await Promise.all(carRequests);
                this.limit = 0;

                this.setState({
                    cars,
                }, () => {
                    this.update();
                });
            } catch (error) {
                if (this.limit < RETRY_LIMIT) {
                    cars = await Promise.all(carRequests);
                    this.limit++;
                    this.update();
                    this.setState({
                        cars,
                    });
                } else {
                    this.setState({ error });
                }
            }
        }, CAR_INFO_TIMEOUT);
    }

    getData() {
        const { userId } = this.props;

        this.props?.getTags?.();

        this.setState({
            isLoading: true,
        }, async () => {
            try {
                const response = await Promise.all([
                    this.request.exec(REQUESTS.GET_USER_INFO, {
                        queryParams: {
                            user_id: userId,
                        },
                    }),
                    this.request.exec(REQUESTS.GET_USER_SESSIONS, {
                        queryParams: {
                            user_id: userId,
                            numdoc: 50,
                        },
                    }),
                    this.request.exec(REQUESTS.GET_SESSIONS_CURRENT, {
                        queryParams: {
                            user_id: userId,
                        },
                    }),
                ]);

                const user = response && response[0];
                const allSessions: SessionsHistoryItem = response[1];
                const sessionsCount = SessionHistoryInfoHandler.getCount.call(allSessions);
                const currentSessions: ISessionInfo[] = [];
                const carIds: string[] = [];

                for (let i = 0; i < sessionsCount; i++) {
                    if (!SessionHistoryInfoHandler.getFinish.call(allSessions, i)) {
                        this.getSession(i, currentSessions, carIds, allSessions);
                    }
                }

                if (!currentSessions.length && SessionHistoryInfoHandler.getSession.call(allSessions)) {
                    this.getSession(0, currentSessions, carIds, allSessions);
                }

                let cars: ICar[] = [];
                const carRequests: Promise<ICar>[] = carIds.filter(car_id => car_id).map((car_id) => {
                    return this.request.exec(REQUESTS.GET_CAR_INFO, {
                        queryParams: {
                            car_id,
                        },
                    });
                });

                try {
                    cars = await Promise.all(carRequests);
                    this.limit = 0;
                    this.update();
                } catch (error) {
                    if (this.limit < RETRY_LIMIT) {
                        cars = await Promise.all(carRequests);
                        this.limit++;
                    } else {
                        this.setState({ error });
                    }
                }

                const SESSION_CURRENT_RESPONSE = 2;
                const sessionCurrentData = response[SESSION_CURRENT_RESPONSE];

                this.setState({
                    carIds,
                    user,
                    sessionCurrentData,
                    currentSessions: {
                        ...response[1],
                        sessions: currentSessions,
                    },
                    cars,
                    isLoading: false,
                });
            } catch (error) {
                this.setState({
                    error,
                    isLoading: false,
                });
            }
        });
    }

    componentDidMount() {
        this.getData();
    }

    componentDidUpdate(prevProps: Readonly<ICCViewProps>) {
        if (prevProps.userId !== this.props.userId) {
            this.getData();
        }
    }

    render() {
        let { error, isLoading, currentSessions, cars, user, sessionCurrentData } = this.state;
        let {} = this.props;
        const count = SessionHistoryInfoHandler.getCount.call(currentSessions);
        if (!user) {
            user = { id: this.props.userId };
        }

        return <>
            {error && <SimpleError error={error}/>}
            {isLoading
                ? <Spin/> : <>
                    <div>
                        {currentSessions && count
                            ? Array.from(Array(count).keys()).map((index: number) => {
                                return (
                                    <React.Fragment key={index}>
                                        <CCViewSession tagsObject={this.props.tagsObject}
                                                       sessionCurrentData={sessionCurrentData}
                                                       sessions={currentSessions}
                                                       index={index}
                                                       car={cars[index]}
                                                       user={user}
                                                       showMap={count === 1}
                                                       getData={this.getData.bind(this)}/>
                                        <hr className={styles.separator}/>
                                    </React.Fragment>
                                );
                            })
                            : (
                                <CCViewEmptySession user={user}/>
                            )
                        }
                    </div>
                </>
            }
        </>;
    }
}
