import * as React from 'react';
import XLSX from 'xlsx';

import { EMPTY_DATA } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import { Collapse } from '../../../ui/Collapse';
import { Input } from '../../../ui/Input';
import { Link } from '../../../ui/Link';
import { NoInformation } from '../../../ui/NoInformation';
import Select, { IOptionInfo } from '../../../ui/Select';
import * as tblStyles from '../../../ui/Table/index.css';
import { TabItem } from '../../../ui/Tabs';
import { Request2 } from '../../../utils/request';
import { IStore } from '../../App/store';
import { SimpleError } from '../../SimpleError';
import Spin from '../../Spin';
import { CLIENTS_CARD_REQUESTS as requestConfigs, REQUESTS } from '../request';
import * as style from './index.css';

type IClientsViewProps = IStore

interface IClientsViewState {
    tabs: TabItem[];
    currentTab: string;
}

export class ClientsView extends React.Component<IClientsViewProps, IClientsViewState> {
    render() {
        const { AdminUser } = this.props;

        return <ClientsSearch AdminUser={AdminUser}/>;
    }
}

enum PERFORMER {
    ALL_PERFORMER = 'allPerformer',
    WITHOUT_PERFORMER = 'withoutPerformer',
    IS_PERFORMER = 'isPerformer',
}

type IClientsSearchProps = IStore

interface IClientsSearchState {
    [x: number]: any;

    has_all_of: string[];
    has_none_of: string[];
    has_one_of: string[];
    selectItems: IOptionInfo[];
    tagsIsLoading: boolean;
    tags: any[];
    error: Error | null;
    clients: any[];
    totalClients: number | null;
    collapse: boolean;
    performer_id: string;
    isLoading: boolean;
    performer: string;
}

export class ClientsSearch extends React.Component<IClientsSearchProps, IClientsSearchState> {
    state: IClientsSearchState = {
        has_all_of: [],
        has_none_of: [],
        has_one_of: [],
        selectItems: [],
        tagsIsLoading: false,
        tags: [],
        error: null,
        clients: [],
        totalClients: null,
        collapse: false,
        performer_id: '',
        isLoading: false,
        performer: PERFORMER.ALL_PERFORMER,
    };

    request = new Request2({ requestConfigs });

    getAllTags() {
        this.setState({
            tagsIsLoading: true,
        }, () => {
            this.request.exec(REQUESTS.GET_TAG_DESCRIPTIONS)
                .then(response => {
                    const tags = response?.records
                        ?.filter((_i: any) => _i?.entities.includes('user') && _i.name);

                    this.setState({
                        tags,
                        tagsIsLoading: false,
                        selectItems: this.getSelectItems(tags),
                    });
                });
        });
    }

    getSelectItems(tags: any[]) {
        const types = tags
            .map((item: any) => {
                return item.type;
            });

        return tags
            .map((item: any) => {
                return {
                    text: item.display_name || item.name,
                    value: item.name,
                    description: item.comment,
                };
            },
            ).concat([...new Set(types)].map(item => {
                return {
                    text: item,
                    value: `@${item}`,
                    description: `Тип: ${item}`,
                };
            }));
    }

    onSelect(key, value) {
        this.setState({
            [key]: value,
        });

        if (key === 'performer_id') {
            this.setState({
                performer: '',
            });
        }
    }

    filter() {
        this.setState({
            isLoading: true,
        }, () => {
            const {
                has_all_of,
                has_none_of,
                has_one_of,
                performer,
                performer_id,
            } = this.state;

            this.request.exec(REQUESTS.GET_TAG_SEARCH, {
                queryParams: {
                    has_all_of: has_all_of.toString(),
                    has_none_of: has_none_of.toString(),
                    has_one_of: has_one_of.toString(),
                    performer_id: performer === PERFORMER.WITHOUT_PERFORMER
                        ? '0' : performer_id
                            ? performer_id
                            : '',
                },
            })
                .then(response => {
                    const clients = response?.results
                        ?.filter((_item: any) => _item.entity_type === 'user')[0]
                        ?.report ?? [];

                    const totalClients = response?.results?.[0]?.total ?? null;

                    this.setState({
                        isLoading: false,
                        error: null,
                        clients,
                        totalClients,
                        collapse: !!clients.length,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    collapseOnClick() {
        this.setState((prev) => ({
            collapse: !prev.collapse,
        }));
    }

    onChangePerformer(type: string) {
        const user_id = this.props.AdminUser?.userId;

        this.setState({
            performer: type,
        }, () => {
            this.setState({
                performer_id: type === PERFORMER.IS_PERFORMER && user_id ? user_id : '',
            });
        });
    }

    download() {
        this.setState({
            isLoading: true,
        }, () => {
            const {
                has_all_of,
                has_none_of,
                has_one_of,
                performer,
                performer_id,
            } = this.state;

            this.request.exec(REQUESTS.GET_TAG_SEARCH, {
                queryParams: {
                    limit: 999999999,
                    has_all_of: has_all_of.toString(),
                    has_none_of: has_none_of.toString(),
                    has_one_of: has_one_of.toString(),
                    performer_id: performer === PERFORMER.WITHOUT_PERFORMER
                        ? '0' : performer_id
                            ? performer_id
                            : '',
                },
            })
                .then(response => {
                    const clients = response?.results
                        ?.filter((_item: any) => _item.entity_type === 'user')[0]
                        ?.report ?? [];

                    const wb = XLSX.utils.book_new();
                    const ws_data: any = [];
                    ws_data.push(
                        [
                            'id',
                            'first_name',
                            'last_name',
                            'pn',
                            'username',
                            'status',
                            'email',
                            'phone',
                        ],
                        ...clients.map((_c: any) => {
                            return [
                                _c.id,
                                _c.first_name,
                                _c.last_name,
                                _c.pn,
                                _c.username,
                                _c.status,
                                _c.setup?.email?.address,
                                _c?.setup?.phone?.number,
                            ];
                        }));

                    const ws = XLSX.utils.aoa_to_sheet(ws_data);
                    XLSX.utils.book_append_sheet(wb, ws, 'Результаты поиска');

                    XLSX.writeFile(wb, `clients-${clients.length}.xlsx`);

                    this.setState({
                        isLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    componentDidMount() {
        this.getAllTags();
    }

    componentWillUnmount() {
        this.request.abort();
    }

    render() {
        const {
            collapse,
            selectItems,
            has_all_of,
            has_none_of,
            has_one_of,
            isLoading,
            error,
            performer_id,
            performer,
            clients,
            totalClients,
        } = this.state;

        return (
            <div>
                <Collapse title={'Параметры поиска'}
                          collapsed={collapse}
                          onClick={this.collapseOnClick.bind(this)}
                          className={style.selectCollapse}>
                    <div className={style.selectContainer}>
                        <Select placeholder={'Наличие тега'}
                                options={selectItems || []}
                                multiSelect
                                onSelect={this.onSelect.bind(this, 'has_all_of')}
                                initialValues={has_all_of}/>
                        <div>
                            {has_all_of.join(', ')}
                        </div>
                        <Select placeholder={'Один из'}
                                options={selectItems || []}
                                multiSelect
                                onSelect={this.onSelect.bind(this, 'has_one_of')}
                                initialValues={has_one_of}/>
                        <div>
                            {has_one_of.join(', ')}
                        </div>
                        <Select placeholder={'Отсутствие тега'}
                                options={selectItems || []}
                                multiSelect
                                onSelect={this.onSelect.bind(this, 'has_none_of')}
                                initialValues={has_none_of}/>
                        <div>
                            {has_none_of.join(', ')}
                        </div>
                        <Select placeholder={'Исполнитель'}
                                options={[{ text: 'Все', value: 'allPerformer' },
                                    { text: 'Пустые', value: 'withoutPerformer' },
                                    { text: 'Мои', value: 'isPerformer' },
                                ]}
                                onSelect={this.onChangePerformer.bind(this)}
                                initialValues={[performer]}/>
                        <Input placeholder={'Id исполнителя'}
                               value={performer_id}
                               onChange={this.onSelect.bind(this, 'performer_id')}/>
                    </div>
                </Collapse>
                <Button className={style.filterButton}
                        disabled={isLoading || !has_none_of.length
                    && !has_all_of.length && !has_one_of.length}
                        onClick={this.filter.bind(this)}>Отфильтровать</Button>
                <Button colorType={ButtonTypes.warning}
                        disabled={isLoading || !has_none_of.length
                        && !has_all_of.length && !has_one_of.length}
                        onClick={this.download.bind(this)}>{'Скачать'}</Button>
                {isLoading ? <Spin/>
                    : error ? <SimpleError error={error}/>
                        : clients.length ?
                            <>
                                <h4>
                                    Нaйдено пользователей {clients.length}/{
                                        totalClients ? totalClients : EMPTY_DATA}
                                </h4>
                                <div className={style.searchTable}>
                                    <table className={tblStyles.table}>
                                        <thead>
                                            <tr>
                                                <th>#</th>
                                                <th>Статус</th>
                                                <th>Логин</th>
                                                <th>Имя</th>
                                                <th>Фамилия</th>
                                                <th>Отчество</th>
                                                <th>Номер телефона</th>
                                                <th>email</th>
                                                <th>Теги</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            {clients.map((client, index) => {
                                                return (
                                                    <tr key={index}>
                                                        <td>{index + 1}</td>
                                                        <td>{client.status}</td>
                                                        <td>
                                                            <Link href={`${location.origin}${location.pathname}#/clients/` +
                                                            `${client.id}/info`}>
                                                                {client.username || client.id}
                                                            </Link>
                                                        </td>
                                                        <td>{client.first_name}</td>
                                                        <td>{client.last_name}</td>
                                                        <td>{client.pn}</td>
                                                        <td>{client?.setup?.phone?.number}</td>
                                                        <td>{client?.setup?.email?.address}</td>
                                                        <td>{client?.tags?.map((t: any) => t.tag).join(', ')}</td>
                                                    </tr>
                                                );
                                            })}
                                        </tbody>
                                    </table>
                                </div>
                            </>
                            : <div className={style.noResults}>
                                <NoInformation/>
                            </div>
                }
            </div>
        );
    }
}
