import React from 'react';

import { EMPTY_DATA } from '../../../constants';
import { BasicUserInfo, UserInfoHandler } from '../../../models/user';
import { Window } from '../../../ui/FullModal';
import { Link } from '../../../ui/Link';
import { UserStatus } from '../../../ui/Status';
import { Request2 } from '../../../utils/request';
import { rubs } from '../../../utils/rubs';
import QueryScheduler from '../../QueryScheduler/QueryScheduler';
import { CenteredSpin } from '../../Spin';
import { CLIENTS_CARD_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { IDevices } from './component';
import style from './index.css';

interface IDeviceSearchModalProps {
    onClose: () => void;
    device: IDevices;
}
interface IDeviceSearchModalState {
    isLoading: boolean;
    searchError: Error | null;
    searchData: BasicUserInfo[] | null;

    isDebtsLoading: boolean;
    debts: Record<string, number | undefined>;
}

export class DeviceSearchModal extends React.Component<IDeviceSearchModalProps, IDeviceSearchModalState> {
    request = new Request2({ requestConfigs });
    state: IDeviceSearchModalState = {
        isLoading: false,
        searchError: null,
        searchData: null,

        isDebtsLoading: false,
        debts: {},
    }

    componentDidMount() {
        this.findAccounts();
    }

    componentDidUpdate(prevProps, prevState: Readonly<IDeviceSearchModalState>) {
        const { searchData } = this.state;

        if (!prevState.searchData && searchData) {
            this.getDebt();
        }
    }

    findAccounts() {
        const { device } = this.props;

        this.setState({ isLoading: true, searchError: null }, () => {
            this.request.exec(REQUESTS.SEARCH, {
                queryParams: {
                    has_all_of: device.id,
                    limit: 50,
                },
            })
                .then(response => {
                    this.setState({
                        isLoading: false,
                        searchData: response?.objects?.users || [],
                    });
                })
                .catch(searchError => {
                    this.setState({
                        searchError,
                        isLoading: false,
                    });
                });
        });
    }

    getDebt() {
        const { searchData } = this.state;
        const userIds = searchData?.map(user => UserInfoHandler.getId.call(user)) ?? [];

        const debtQueue = userIds.map((user_id) => {
            return this.request.exec.bind(this.request, REQUESTS.GET_USER_INFO,
                { queryParams: { user_id } },
            );
        });

        const qs = new QueryScheduler({
            queue : debtQueue,
            limit: 100,
            onProgress: () => {},
            onSuccess: this.onSuccess.bind(this),
        });

        this.setState({ isDebtsLoading: true }, () => {
            qs.run();
        });
    }

    onSuccess(info) {
        const debts = {};

        info.success.forEach(el => {
            debts[el.id] = el.billing?.debt;
        });

        this.setState({ debts, isDebtsLoading: false });
    }

    render() {
        const { onClose } = this.props;
        const { searchError, isLoading, searchData, isDebtsLoading, debts } = this.state;
        const UIH = UserInfoHandler;

        return <Window onClose={onClose} title={'Поиск по устройству'} error={searchError}>
            <div className={style.devices_modal_wrapper}>
                {isLoading
                    ? <CenteredSpin/>
                    : searchData?.length
                        ? <table className={style.users_table}>
                            <thead>
                                <tr>
                                    <td>Пользователь</td>
                                    <td>Телефон</td>
                                    <td>Статус</td>
                                    <td>Долг</td>
                                </tr>
                            </thead>
                            <tbody>
                                {searchData.map(userInfo => {
                                    const userId = UIH.getId.call(userInfo);
                                    const debt = debts[userId];

                                    return <tr key={userId}>
                                        <td>
                                            <Link href={`#/clients/${userId}/info`} target={'_blank'}>
                                                {UIH.getPrintName.call(userInfo)}
                                            </Link>
                                        </td>
                                        <td>{UIH.getPhone.call(userInfo)}</td>
                                        <td>
                                            <UserStatus text={UIH.getStatus.call(userInfo)}/>
                                        </td>
                                        <td>
                                            {isDebtsLoading
                                                ? <CenteredSpin size={'s'}/>
                                                : Object.keys(debts).length
                                                    ? <div className={`${debt && debt > 0 ? style.debt : ''}`}>
                                                        {debt ? rubs(debt) : EMPTY_DATA}
                                                    </div>
                                                    : <div className={style.empty_debt}>
                                                        Не получены данные
                                                    </div>
                                            }
                                        </td>
                                    </tr>;
                                })}
                            </tbody>
                        </table>
                        : <h3>
                                Не удалось найти пользователей
                        </h3>
                }
            </div>
        </Window>;
    }
}
