import React from 'react';

import { ONE_SECOND } from '../../../constants';
import { Button, ButtonTypes } from '../../../ui/Button';
import FormatDate from '../../../ui/FormatDate';
import { IPicture } from '../../../ui/PictureBlocksViewer';
import { UserStatuses } from '../../../ui/Status';
import { Switcher } from '../../../ui/Switcher';
import { Request2 } from '../../../utils/request';
import { PictureGallery } from '../../PictureGallery';
import TagModal from '../../TagModal';
import { OBJECT_TYPES } from '../../TagModal/component';
import { CLIENTS_CARD_REQUESTS as requestConfigs, REQUESTS } from '../request';
import { DocStatus, IPhoto } from './component';
import { CGI_SEPARATOR, MEDIA_ITEMS, PHOTO_SHOW_MODE, TYPE_ITEMS, VIDEO_SHOW_MODE } from './constants';
import * as style from './index.css';

interface IMediaGalleryProps {
    onCurrentPhotosChange: (value: string) => void;
    currentPhotoType: string;
    BlockRules: Record<string, boolean>;
    showVideo: boolean;
    onDocTypeChange: (value: string) => void;
    userInfo: Record<string, any>;
    openRegisterModal: () => void;
    currentPhotos: IPhoto[];
    onPictureChange: (index: number) => void;
    currentPhotoIndex: number | null;
}

interface IMediaGalleryState {
    ReRequestForChatIsOpen: boolean;
    photos: Record<string,IPicture[]>;
    videos: Record<string,IPicture[]>;
    errors: Record<string, (Error | null)[]>;
}

export default class MediaGallery extends React.Component<IMediaGalleryProps, IMediaGalleryState> {
    state: IMediaGalleryState = {
        ReRequestForChatIsOpen: false,
        photos: {},
        videos: {},
        errors: {},
    };
    request = new Request2({ requestConfigs });

    componentDidMount() {
        this.setInitialMedia();
    }

    componentDidUpdate(prevProps: Readonly<IMediaGalleryProps>) {
        const { showVideo, currentPhotoIndex, currentPhotoType } = this.props;
        const currentArray = this.getCurrentMedia();

        if (currentPhotoType && !currentArray) {
            this.setInitialMedia();
        }

        if (!currentArray?.[currentPhotoIndex || 0]) {
            this.preparePhotos(showVideo);
        }
    }

    getCurrentMedia() {
        const { showVideo, currentPhotoType } = this.props;
        const { videos, photos } = this.state;

        return showVideo ? videos[currentPhotoType] : photos[currentPhotoType];
    }

    setInitialMedia() {
        const { currentPhotos, currentPhotoType } = this.props;

        this.setState({
            photos: {
                [currentPhotoType]: new Array(currentPhotos.length),
            },
            videos: {
                [currentPhotoType]: new Array(currentPhotos.length),
            },
        });
    }

    preparePhotos(showVideo?: boolean) {
        const { currentPhotos, currentPhotoType, userInfo, currentPhotoIndex = 0 } = this.props;
        const { errors: stateErrors } = this.state;
        const currentArray = this.getCurrentMedia()?.slice() || [];

        this.request.exec(REQUESTS.GET_USER_DOC_DATA, {
            queryParams: {
                user_id: userInfo?.id,
                photo_id: currentPhotos[currentPhotoIndex || 0]?.id,
                background_video: showVideo,
            },
            blob: true,
        })
            .then(response => {
                const { videos, photos } = this.state;
                const { currentPhotoIndex } = this.props;
                const URL = window.URL || window.webkitURL;

                const currentErrors = stateErrors[currentPhotoType] || [];
                currentErrors[currentPhotoIndex || 0] = null;
                const errors = { ...stateErrors, [currentPhotoType]: currentErrors };
                currentArray[currentPhotoIndex || 0] = { link: URL.createObjectURL(response), info: {} };

                if (showVideo) {
                    this.setState({
                        videos: {
                            ...videos,
                            [currentPhotoType]: currentArray,
                        },
                    });
                } else {
                    this.setState({
                        photos: {
                            ...photos,
                            [currentPhotoType]: currentArray,
                        },
                        errors,
                    });
                }
            }).catch((error) => {
                currentArray[currentPhotoIndex || 0] = { link: 'error', info: {} };

                if (!showVideo) {
                    const { photos } = this.state;
                    const currentErrors = stateErrors[currentPhotoType] || [];
                    currentErrors[currentPhotoIndex || 0] = error;
                    const errors = { ...stateErrors, [currentPhotoType]: currentErrors };
                    this.setState({ errors, photos: {
                        ...photos,
                        [currentPhotoType]: currentArray,
                    } });
                } else {
                    const { videos } = this.state;

                    this.setState({ videos: {
                        ...videos,
                        [currentPhotoType]: currentArray,
                    } });
                }
            });
    }

    openReRequestForChat(ReRequestForChatIsOpen) {
        this.setState({ ReRequestForChatIsOpen });
    }

    closeCurrentPhotos() {
        location.hash = location.hash.split(CGI_SEPARATOR)[0];
    }

    getGalleryControls() {
        const {
            onCurrentPhotosChange,
            currentPhotoType,
            BlockRules,
            showVideo,
            onDocTypeChange,
            userInfo,
            openRegisterModal,
            currentPhotos,
            onPictureChange,
            currentPhotoIndex,
        } = this.props;

        return <div className={style.picture_gallery_controls}>
            <div className={style.photo_type_switch}>
                <Switcher items={TYPE_ITEMS}
                          onChange={onCurrentPhotosChange.bind(this)}
                          active={currentPhotoType}/>
            </div>
            {BlockRules?.SecretX
                ? <div className={style.photo_type_switch}>
                    <Switcher items={MEDIA_ITEMS}
                              onChange={onDocTypeChange.bind(this)}
                              active={showVideo ? VIDEO_SHOW_MODE : PHOTO_SHOW_MODE}/>
                </div>
                : null
            }
            <div className={style.button_container}>
                {userInfo && [
                    UserStatuses.REJECTED,
                    UserStatuses.ON_BOARDING,
                    UserStatuses.SCREENING,
                    UserStatuses.FAST_REG,
                ].includes(userInfo.status) && BlockRules?.RegisterUser
                    ? <Button className={style.register_button}
                              colorType={ButtonTypes.positive}
                              onClick={openRegisterModal.bind(this)}>Зарегистрировать</Button>
                    : null
                }
            </div>
            <table className={style.photo_checks}>
                <tbody>
                    {currentPhotos?.map((photo, index) => {
                        const onClickHandler = onPictureChange.bind(this, index);

                        return <tr onClick={onClickHandler}
                                   key={photo ? photo.uri : index}
                                   className={`${currentPhotoIndex === index
                                       ? style.current_photo_check
                                       : ''} ${photo.verified_at ? '' : style.no_info}`}>
                            <td>
                                <FormatDate value={photo?.verified_at * ONE_SECOND}/>
                            </td>
                            <td>
                                <DocStatus type={photo?.verification_status}/>
                            </td>
                        </tr>;
                    })}

                </tbody>
            </table>
            <Button colorType={ButtonTypes.positive}
                    onClick={this.openReRequestForChat.bind(this, true)}>
                Дозапросить для чата
            </Button>
        </div>;
    }

    render() {
        const {
            showVideo,
            currentPhotoIndex,
            currentPhotoType,
            onPictureChange,
            userInfo,
        } = this.props;

        const { ReRequestForChatIsOpen, errors } = this.state;

        const galleryControls = this.getGalleryControls();

        return <>
            <PictureGallery video={showVideo}
                            controls={galleryControls}
                            initialIndex={currentPhotoIndex || 0}
                            onChange={onPictureChange.bind(this)}
                            pictures={this.getCurrentMedia()}
                            errors={errors[currentPhotoType]}
                            onClose={this.closeCurrentPhotos.bind(this)}/>

            {ReRequestForChatIsOpen
                ? <TagModal objectId={{ type: OBJECT_TYPES.USER, id: userInfo?.id }}
                            initialData={{ tag: 'documents_reask_tag' }}
                            withoutPropositions={true}
                            onClose={this.openReRequestForChat.bind(this, false)}/>
                : null
            }
        </>;
    }
}
