import React from 'react';

import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import { Button } from '../../../../ui/Button';
import FormatDate from '../../../../ui/FormatDate';
import { Window } from '../../../../ui/FullModal';
import { JsonModal } from '../../../../ui/FullModal/JsonModal';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import { Request2 } from '../../../../utils/request';
import { Copy } from '../../../Copy';
import Spin from '../../../Spin';
import TagModal from '../../../TagModal';
import { OBJECT_TYPES } from '../../../TagModal/component';
import style from '../index.css';
import { ICheckInfo } from '../types';
import { DOCUMENTS_CHECK_REQUESTS, REQUESTS } from './requests';

interface ICheckModalProps {
    user_id: string;
    check: Record<string, any>;
    onClose: () => void;
}

interface ICheckModalState {
    isLoading: boolean;
    error: Error | null;
    data: ICheckInfo[] | null;
    jsonElement: ICheckInfo | null;
    isTagModalOpened: boolean;
}

export default class CheckModal extends React.Component<ICheckModalProps, ICheckModalState> {
    state: ICheckModalState = {
        isLoading: false,
        error: null,
        data: null,
        jsonElement: null,
        isTagModalOpened: false,
    };
    request = new Request2({ requestConfigs: DOCUMENTS_CHECK_REQUESTS });

    componentDidMount() {
        this.getData();
    }

    getData() {
        const { user_id, check } = this.props;

        this.setState({ isLoading: true, error: null }, () => {
            this.request.exec(REQUESTS.GET_STATUS, {
                body: { user_id, type: (check?.id || check?.type) },
            })
                .then(data => {
                    this.setState({
                        data,
                        isLoading: false,
                    });
                })
                .catch(error => {
                    this.setState({
                        error,
                        isLoading: false,
                    });
                });
        });
    }

    openJSONModal(jsonElement) {
        this.setState({
            jsonElement,
        });
    }

    openTagModal(isTagModalOpened: boolean) {
        this.setState({
            isTagModalOpened,
        });
    }

    render() {
        const { isLoading, error, data = [], jsonElement, isTagModalOpened } = this.state;
        const { onClose, check, user_id } = this.props;

        return <>
            <Window title={`История проверки ${check?.title ?? check?.type}`}
                    error={error}
                    onClose={onClose.bind(this)}>
                {isLoading
                    ? <Spin/>
                    : data?.length
                        ? <>
                            <table className={style.checks_table}>
                                <tr>
                                    <td>Дата</td>
                                    <td>Статус</td>
                                    <td/>
                                    <td>Пользователь</td>
                                </tr>

                                {data.sort((a, b) => b.timestamp - a.timestamp)
                                    .map((el, ind) => {
                                        const status = `${style.check_item_common}
                                        ${style[`${(!el.status || el.status === 'new') ? 'NONE' : el.status}_status`]}`;

                                        return <tr key={el.event_id || ind}>
                                            <td>
                                                <FormatDate value={el.timestamp * ONE_SECOND}/>
                                            </td>

                                            <td>
                                                <div className={status}>
                                                    {el.status}
                                                </div>
                                            </td>

                                            <td>
                                                <div>Secret_id:
                                                    <Copy text={el.secret_id}>
                                                        {el.secret_id || EMPTY_DATA}
                                                    </Copy>
                                                </div>
                                                <div>Worker_id:
                                                    <Copy text={el.worker_id}>
                                                        {el.worker_id || EMPTY_DATA}
                                                    </Copy>
                                                </div>
                                            </td>

                                            <td>
                                                <Link href={`#/clients/${el.user_id}/info`}>
                                                    {el.login || el.user_id}
                                                </Link>
                                            </td>

                                            <td>
                                                <Link onClick={this.openJSONModal.bind(this, el)}>
                                                JSON
                                                </Link>
                                            </td>
                                        </tr>;
                                    })}
                            </table>

                            <div className={style.group_act_controls}>
                                {check?.actions?.[0]?.tagName
                                    ? <Button onClick={this.openTagModal.bind(this, true)}>
                                        Перезапросить
                                    </Button>
                                    : null
                                }
                            </div>
                        </>
                        : <NoInformation/>
                }
            </Window>

            {jsonElement
                ? <JsonModal obj={jsonElement} onClose={() => this.setState({ jsonElement: null })}/>
                : null
            }

            {isTagModalOpened
                ? <TagModal initialData={{ tag: check?.actions?.[0]?.tagName }}
                            title={`Перезапрос ${check.title}`}
                            objectId={{ type: OBJECT_TYPES.USER, id: user_id }}
                            onClose={this.openTagModal.bind(this, false)}/>
                : null
            }
        </>;
    }
}
