import React from 'react';

import { EMPTY_DATA } from '../../../../constants';
import { Collapse2 } from '../../../../ui/Collapse2';
import { Request2 } from '../../../../utils/request';
import { GroupActions } from '../../../RegistrationTimeline/GroupActions';
import { SimpleError } from '../../../SimpleError';
import Spin from '../../../Spin';
import style from '../index.css';
import { ICheckInfoItem, ICheckItem } from '../types';
import CheckModal from './CheckModal';
import { DOCUMENTS_CHECK_REQUESTS, REQUESTS } from './requests';

interface IReissueDocChecksProps {
    userId: string;
    regFlowData: Record<string, any>;
    registration_flow: Record<string, any>;
}

interface IReissueDocChecksState {
    isChecksLoading: boolean;
    checksError: Error | null;
    checksData: ICheckItem[] | null;
    translates: Record<string, string>;
    existedSettings: Record<string, ICheckInfoItem>;
    checkForHistory: ICheckInfoItem | ICheckItem | null;
}

export default class ReissueDocChecks extends React.Component<IReissueDocChecksProps, IReissueDocChecksState> {
    state: IReissueDocChecksState = {
        isChecksLoading: false,
        checksError: null,
        checksData: null,
        translates: {},
        existedSettings: {},
        checkForHistory: null,
    };
    request = new Request2({ requestConfigs: DOCUMENTS_CHECK_REQUESTS });

    componentDidMount() {
        this.getChecks();
    }

    componentDidUpdate(prevProps: Readonly<IReissueDocChecksProps>): void {
        const { userId } = this.props;
        if (userId !== prevProps.userId) {
            this.getChecks();
        }
    }

    getChecks() {
        const { userId } = this.props;

        this.setState({ isChecksLoading: true, checksError: null }, () => {
            this.request.exec(REQUESTS.GET_CHECKS, {
                body: { user_id: userId },
            })
                .then(checksData => {
                    const { registration_flow } = this.props;
                    const sortedChecksData = (checksData ?? []).map((el) => {
                        return { ...el, order: registration_flow[el.type]?.order ?? 0 };
                    }).sort((a, b) => a.order - b.order);
                    const { translates, existedSettings } = this.getTranslates(sortedChecksData);

                    this.setState({
                        translates,
                        existedSettings,
                        isChecksLoading: false,
                        checksData: sortedChecksData,
                    });
                })
                .catch(checksError => {
                    this.setState({
                        isChecksLoading: false,
                        checksError,
                    });
                });
        });
    }

    getTranslates(checksData) {
        const { regFlowData } = this.props;
        const data = JSON.parse(regFlowData?.setting_value) ?? [];
        const translates = {};
        const existedSettings = {};

        data.forEach(el => {
            translates[el.id] = el.title;

            const checkItem = checksData?.find(check => check.type === el.id);
            if (checkItem) {
                existedSettings[el.id] = el;
            }
        });

        return { translates, existedSettings };
    }

    openCheck(check) {
        const { existedSettings } = this.state;
        const checkForHistory = existedSettings[check.type] || check;

        this.setState({
            checkForHistory,
        });
    }

    closeCheck() {
        this.setState({
            checkForHistory: null,
        });
    }

    render() {
        const { isChecksLoading, checksError, checksData, translates, checkForHistory, existedSettings } = this.state;
        const { userId } = this.props;

        return <div>
            {isChecksLoading
                ? <Spin/>
                : checksError
                    ? <Collapse2 title={'Хьюстон, у нас проблемы'} headerClassname={style.error_collapse}>
                        <SimpleError error={checksError}/>
                    </Collapse2>
                    : <div>
                        <h4>Проверки (новый процесс) {!checksData?.length && EMPTY_DATA}</h4>
                        <div>
                            {checksData?.map((check, ind) => {
                                const key = `${ind}-${check.secret_id}` || ind;
                                const additionalStyle = `${(!check.status || check.status === 'new')
                                    ? 'NONE' : check.status}_status`;

                                return <div key={key}
                                            onClick={this.openCheck.bind(this, check)}
                                            className={`${style.check_item_common} ${style[additionalStyle]}`}>
                                    {translates[check.type] || check.type}
                                </div>;
                            })
                            }
                        </div>
                        {checksData?.length
                            ? <div className={style.group_act_controls}>
                                <GroupActions uniqueTag={checkForHistory?.type} flowItem={existedSettings}/>
                            </div>
                            : null
                        }
                    </div>
            }

            {checkForHistory
                ? <CheckModal user_id={userId}
                              check={checkForHistory}
                              onClose={this.closeCheck.bind(this)}/>
                : null
            }
        </div>;
    }
}
