import React from 'react';
import { Route, Switch } from 'react-router';

import { Dict } from '../../../../../types';
import { EMPTY_DATA, ONE_SECOND } from '../../../../constants';
import FormatDate from '../../../../ui/FormatDate';
import { Link } from '../../../../ui/Link';
import { NoInformation } from '../../../../ui/NoInformation';
import { IPicture } from '../../../../ui/PictureBlocksViewer';
import { ISwitcherItem, Switcher } from '../../../../ui/Switcher';
import { isObjectEqual } from '../../../../utils/isObjectEqual';
import { Request2 } from '../../../../utils/request';
import Img from '../../../Img';
import Spin from '../../../Spin';
import { CLIENTS_CARD_REQUESTS as requestConfigs, REQUESTS } from '../../request';
import { DOCUMENTS_KEYS } from '../constants';
import * as style from '../index.css';
import SelfiesGallery from './SelfiesGallery';

enum SwitcherOptions {
    photo = 'p',
    video = 'x'
}

const SELFIE_INDEX_IDENTIFIER = 'selfie=';
const VIEW_OPTION_IDENTIFIER = '&option=';

interface IDocumentSelfiesProps {
    location: Dict<any>;
    selfies: Dict<any>[];
    userId: string;
    secretXRule: boolean;
}

interface IDocumentSelfiesState {
    selfies: Dict<any>;
    showImgIndex: number | null;
    chosenSwitcherOption: SwitcherOptions;
    photoErrors: Error[] | null;
    isLoading: boolean;
}

export default class DocumentSelfies extends React.Component<IDocumentSelfiesProps, IDocumentSelfiesState> {
    state: IDocumentSelfiesState = {
        selfies: { photoItems: [] },
        showImgIndex: null,
        chosenSwitcherOption: SwitcherOptions.photo,
        photoErrors: null,
        isLoading: true,
    };

    request = new Request2({
        requestConfigs,
    });
    defaultLocation: string;
    switcherOptions: ISwitcherItem[] = [
        { text: 'Фото', value: SwitcherOptions.photo },
        { text: 'Видео', value: SwitcherOptions.video },
    ];

    componentDidMount(): void {
        this.checkLocation();
    }

    componentDidUpdate(
        prevProps: Readonly<IDocumentSelfiesProps>,
        prevState: Readonly<IDocumentSelfiesState>,
        snapshot?: any,
    ): void {
        if (this.props.location.pathname !== prevProps.location.pathname
            || this.props.location.pathname !== this.defaultLocation) {
            this.checkLocation();
        }

        if (prevProps.userId !== this.props.userId
            || (this.props.selfies && !isObjectEqual(prevProps.selfies, this.props.selfies))) {
            this.setDefaultLocation();
            this.getSelfies();
            this.checkLocation();
        }
    }

    getSelfies(selfie = this.props.selfies?.[0], index = 0) {
        const selfies: Record<string, any> = { ...this.state.selfies };

        if (this.props.selfies?.length) {
            selfie?.id && this.request.exec(REQUESTS.GET_USER_DOC_DATA, {
                queryParams: {
                    user_id: this.props.userId,
                    photo_id: selfie.id,
                },
                blob: true,
            })
                .then(photoBlob => {
                    const URL = window.URL || window.webkitURL;
                    const blobLink = URL.createObjectURL(photoBlob);

                    selfies.photoItems.push({ link: blobLink, info: {} });

                    this.setState({ selfies }, () => {
                        if (index < this.props.selfies?.length - 1) {
                            this.getSelfies(this.props.selfies?.[index + 1], index + 1);
                        } else {
                            this.setState({ isLoading: false });
                        }
                    });
                })
                .catch(photoError => {
                    const errors = this.state.photoErrors ? [...this.state.photoErrors] : [];
                    errors[index] = photoError;
                    selfies.photoItems.push({ link: 'error', info: { retry: true } });

                    this.setState({ photoErrors: errors, selfies }, () => {
                        if (index < this.props.selfies?.length - 1) {
                            this.getSelfies(this.props.selfies?.[index + 1], index + 1);
                        } else {
                            this.setState({ isLoading: false });
                        }
                    });
                });
        } else {
            this.setState({ isLoading: false });
        }
    }

    setDefaultLocation() {
        this.defaultLocation = this.props.userId ? `/clients/${this.props.userId}/docs/` : '';
    }

    changeLocation(index: number | null, switcherOption?: SwitcherOptions) {
        const viewOption = this.props.secretXRule
            ? switcherOption
                ? switcherOption
                : this.state.chosenSwitcherOption
            : SwitcherOptions.photo;

        location.hash = index !== null
            ? `${this.defaultLocation}${SELFIE_INDEX_IDENTIFIER}${index + 1}${VIEW_OPTION_IDENTIFIER}${viewOption}`
            : this.defaultLocation;
    }

    checkLocation() {
        if (location.hash !== this.defaultLocation) {
            const showImgIndex = this.findIndex();
            const chosenSwitcherOption = this.findSwitcherOption();

            if (showImgIndex !== this.state.showImgIndex
                || chosenSwitcherOption !== this.state.chosenSwitcherOption) {
                this.setState({ showImgIndex, chosenSwitcherOption });
            }
        }
    }

    findIndex(): number | null {
        const startIndex = location.hash.indexOf(SELFIE_INDEX_IDENTIFIER);
        const finishIndex = location.hash.indexOf(VIEW_OPTION_IDENTIFIER);

        return startIndex !== -1
            ? Number(location.hash.substring(startIndex + SELFIE_INDEX_IDENTIFIER.length, finishIndex)) - 1
            : null;
    }

    findSwitcherOption(): SwitcherOptions {
        if (this.props.secretXRule) {
            const startIndex = location.hash.indexOf(VIEW_OPTION_IDENTIFIER);

            return startIndex !== -1
                ? location.hash.substring(startIndex + VIEW_OPTION_IDENTIFIER.length) as SwitcherOptions
                : SwitcherOptions.photo;
        }

        return SwitcherOptions.photo;

    }

    onSwitcherChange(switcherOption: SwitcherOptions) {
        this.changeLocation(this.state.showImgIndex, switcherOption);
    }

    showImg(showImgIndex: number | null) {
        this.changeLocation(showImgIndex);
    }

    getListItemPhoto(item: IPicture, index: number | null) {
        return <Link onClick={this.showImg.bind(this, index)}>
            <Img className={style.photoContainer}
                 src={item.link}
                 error={this.state.photoErrors?.[index || 0]}
                 thumbnail={true}>
            </Img>
        </Link>;
    }

    getControls() {
        if (!this.props.secretXRule) {
            return;
        }

        const { showImgIndex } = this.state;
        const { selfies } = this.props;
        const item = showImgIndex !== null ? selfies[showImgIndex] : {};
        const date = item?.submitted_at
            ? <FormatDate value={item.submitted_at * ONE_SECOND} withSecond={true}/>
            : EMPTY_DATA;

        return <div>
            <div className={style.sidebar_date}>
                {date}
            </div>

            {item?.uri && item?.background_video_uri
                ? <div className={style.switcher}>
                    <Switcher items={this.switcherOptions}
                              active={this.state.chosenSwitcherOption}
                              onChange={this.onSwitcherChange.bind(this)}/>
                </div>
                : null
            }
        </div>;
    }

    render() {
        const { selfies, showImgIndex, chosenSwitcherOption, isLoading } = this.state;

        return <div className={`${style.user_doc_item_data} ${style.selfies_block}`}>
            <div className={style.selfies_header}>
                <h2>{DOCUMENTS_KEYS.selfies.display_name}</h2>
                {isLoading && <Spin size={'x'}/>}
            </div>

            {this.props.selfies?.length
                ? <div className={style.photos_container}>
                    {this.props.selfies.map((photo, index) => {
                        const item = selfies?.photoItems?.[index] || {};
                        const href = `#${this.defaultLocation}${SELFIE_INDEX_IDENTIFIER}${encodeURIComponent(index + 1)}${VIEW_OPTION_IDENTIFIER}${encodeURIComponent(chosenSwitcherOption)}`;

                        return <Link href={href} key={index} className={style.photo_item}>
                            {item?.link
                                ? this.getListItemPhoto(item, index)
                                : null
                            }
                        </Link>;
                    })}
                </div>
                : !isLoading ? <NoInformation/> : null
            }

            <Switch>
                <Route path={`${this.defaultLocation}selfie=:selfie_index&option=:view_option`}
                       render={() => {
                           const isPhotoOption = chosenSwitcherOption === SwitcherOptions.photo;

                           return <div>
                               {showImgIndex !== null
                                   ? <SelfiesGallery currentPhotoIndex={showImgIndex}
                                                     userId={this.props.userId}
                                                     onPictureChange={this.showImg.bind(this)}
                                                     getControls={this.getControls.bind(this)}
                                                     showVideo={!isPhotoOption}
                                                     modifiedPhotos={selfies.photoItems}
                                                     selfies={this.props.selfies}/>
                                   : null
                               }
                           </div>;
                       }
                       }/>
            </Switch>
        </div>;
    }
}
